<?php

namespace App\Console\Commands;

use App\Notifications\MatchesMissingPlayersNotification;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Notification;
use Modules\Match\Models\Matche;
use Modules\User\Models\User;

class NotifyMatchesMissingPlayers extends Command
{
    protected $signature = 'notify:matches-missing-players';
    protected $description = 'Notify users about nearby matches missing players';

    public function handle()
    {
        $matches = Matche::where('date', '>=', Carbon::now()->format('Y-m-d'))
            ->where('date', '<=', Carbon::now()->addDay()->format('Y-m-d'))
            ->active()
            ->with(['playground', 'teamPlayers'])
            ->get();

        foreach ($matches as $match) {
            // Check if match is full
            $maxPlayers = $match->players_per_side * 2;
            if ($match->teamPlayers->count() >= $maxPlayers) {
                continue;
            }

            $playground = $match->playground;
            if (!$playground || !$playground->Latitude || !$playground->Longitude) {
                continue;
            }

            // Find users within 20km
            $lat = $playground->Latitude;
            $lng = $playground->Longitude;

            $users = User::select('*')
                ->selectRaw("(6371 * acos(cos(radians(?)) * cos(radians(latitude)) * cos(radians(longitude) - radians(?)) + sin(radians(?)) * sin(radians(latitude)))) AS distance", [$lat, $lng, $lat])
                ->having('distance', '<=', 20)
                ->get();

            $usersToNotify = $users->filter(function ($user) use ($match) {
                // Don't notify if already joined
                if ($match->teamPlayers->contains($user->id)) {
                    return false;
                }
                // Check notification setting
                return $user->wantsNotification('Matches Missing Players');
            });

            if ($usersToNotify->count() > 0) {
                Notification::send($usersToNotify, new MatchesMissingPlayersNotification($match));
                $this->info("Sent missing players alert for match ID: {$match->id} to {$usersToNotify->count()} users.");
            }
        }
    }
}
