<?php

namespace Modules\Match\Services;

use App\Services\Services;
use Carbon\Carbon;
use Modules\Match\Models\Matche;

use Modules\Playground\Models\Playground;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\DB;

class MatchService extends Services
{

    private $colums = ['players_per_side', 'gender', 'status', 'price', 'date', 'start_time', 'end_time'];


    public function paginate()
    {
        $column = request('sort_by', 'id');
        $order = request('sort_order') ?: 'desc';
        $players_per_side = request('players_per_side', null);
        $price = request('price', null);
        $start_time = request('start_time', null);
        $end_time = request('end_time', null);
        $gender = request('gender', null);
        $startprice = request('startprice', null);
        $endprice = request('endprice', null);
        $query = Matche::query()->with('user', 'playground')
            ->when(config('context.category_id'), function ($q, $catId) {
                $q->whereHas('playground', function ($q) use ($catId) {
                    $q->where('Category_ID', $catId);
                });
            })
            ->when($players_per_side, function ($q, $players_per_side) {
                $q->where('players_per_side', 'LIKE', "%$players_per_side%");
            })
            ->when($price, function ($q, $price) {
                $q->where('price', 'LIKE', "%$price%");
            })
            ->when($start_time, function ($q, $start_time) {
                $q->where('start_time', 'LIKE', "%$start_time%");
            })
            ->when($gender, function ($q, $gender) {
                $q->where('gender', '=', $gender);
            })->when($startprice, function ($q, $startprice) {
                $q->where('price', '>=', $startprice);
            })
            ->when($endprice, function ($q, $endprice) {
                $q->where('price', '<=', $endprice);
            })
            ->when($column && array_key_exists($column, $this->colums), function ($q) use ($column, $order) {
                $q->orderBy($this->colums[$column], $order);
            });

        return $query->paginate(request('limit', 15));
    }

    public function store(array $data)
    {
        $playground = Playground::find($data['playground_id']);
        if (!$playground) {
            $this->setError(__('messages.Playground not found'));
            return false;
        }

        if ($playground && !empty($playground->slots)) {
            if (!in_array($data['players_per_side'], $playground->slots)) {
                throw ValidationException::withMessages([
                    'players_per_side' => [__('messages.Invalid players per side for this playground')],
                ]);
            }
        }
        $date = Carbon::parse($data['date'])->format('Y-m-d');
        $newStart = $data['start_time'];
        $newEnd = $data['end_time'];

        $isBooked = Matche::where('playground_id', $data['playground_id'])
            ->where('date', $date)
            ->where(function ($query) use ($newStart, $newEnd) {
                $query->where('start_time', '<', $newEnd)
                    ->where('end_time', '>', $newStart);
            })
            ->exists();

        if ($isBooked) {
            throw ValidationException::withMessages([
                'date' => [__('messages.Playground is booked at this time')],
            ]);
        }
        $data['organizer_user_id'] = auth('api')->id();
        return Matche::create($data);
    }


    public function update(array $data, $id)
    {
        $match = Matche::find($id);
        if (!$match) {
            $this->setError(__('messages.Match not found'));
            return false;
        }
        $match->update($data);
        return $match;
    }

    public function destroy($id)
    {
        $match = Matche::find($id);
        if (!$match) {
            $this->setError(__('messages.Match not found'));
            return false;
        }
        $match->delete();
        return $match;
    }

    public function show($id)
    {
        $match = Matche::with('user', 'playground', 'teamPlayers.location')
            ->withCount([
                'teamPlayers',
                'teamPlayers as white_team_players_count' => function ($query) {
                    $query->where('team_side', 'White');
                },
                'teamPlayers as dark_team_players_count' => function ($query) {
                    $query->where('team_side', 'Dark');
                }
            ])
            ->find($id);

        if (!$match) {
            $this->setError(__('messages.Match not found'));
            return false;
        }
        return $match;
    }

    public function upcommingmatches($playground_id)
    {
        $matches = Matche::where('playground_id', $playground_id)->where('date', '>', now())->paginate(request('limit', 15));
        return $matches;
    }

    public function joinMatch($matchId, $teamSide)
    {
        $user = auth('api')->user();
        if ($user->role->name != 'player') {
            throw ValidationException::withMessages([
                'role' => [__('messages.You are not allowed to join this match')],
            ]);
        }
        $match = Matche::find($matchId);
        if (!$match) {
            $this->setError(__('messages.Match not found'));
            return false;
        }

        return DB::transaction(function () use ($matchId, $teamSide, $user) {
            // Lock the match row for update to prevent race conditions
            $match = Matche::where('id', $matchId)->lockForUpdate()->firstOrFail();

            if ($match->gender !== $user->gender) {
                throw ValidationException::withMessages([
                    'gender' => [__('messages.Invalid gender')],
                ]);
            }
            // Validate team side
            if (!in_array($teamSide, ['White', 'Dark'])) {
                throw ValidationException::withMessages([
                    'team_side' => [__('messages.Invalid team side')],
                ]);
            }

            // Check if user already joined (any side)
            if ($match->teamPlayers()->where('user_id', $user->id)->exists()) {
                throw ValidationException::withMessages([
                    'match_id' => [__('messages.You have already joined this match')],
                ]);
            }

            // Check capacity for the specific side
            $currentCount = $match->teamPlayers()->wherePivot('team_side', $teamSide)->count();
            if ($currentCount >= $match->players_per_side) {
                throw ValidationException::withMessages([
                    'team_side' => [__('messages.This team is full')],
                ]);
            }

            $match->teamPlayers()->attach($user->id, ['team_side' => $teamSide]);

            $owner = $match->organizer;
            if ($owner && $owner->id !== $user->id && $owner->wantsNotification('Players Joined My Match')) {
                $owner->notify(new \Modules\Match\Notifications\PlayerJoinedMatchNotification($match, $user));
            }

            return $match;
        });
    }



    public function getNearest($lat, $lng)
    {
        $catId = config('context.category_id');

        $query = Matche::active()
            ->with(['playground', 'user', 'teamPlayers.location'])
            ->withCount([
                'teamPlayers',
                'teamPlayers as blue_team_players_count' => function ($query) {
                    $query->where('team_side', 'blue');
                },
                'teamPlayers as red_team_players_count' => function ($query) {
                    $query->where('team_side', 'red');
                }
            ]);

        if ($catId) {
            $query->whereHas('playground', function ($q) use ($catId) {
                $q->where('Category_ID', $catId);
            });
        }

        $matches = $query->nearest($lat, $lng)
            ->take(5)
            ->get();

        if ($matches->isEmpty()) {
            // Fallback: Get latest matches but still calculate distance for display
            $haversine = "(6371 * acos(cos(radians(?)) 
                         * cos(radians(playgrounds.Latitude)) 
                         * cos(radians(playgrounds.Longitude) - radians(?)) 
                         + sin(radians(?)) 
                         * sin(radians(playgrounds.Latitude))))";

            $fallbackQuery = Matche::active()
                ->select('matches.*')
                ->join('playgrounds', 'matches.playground_id', '=', 'playgrounds.id')
                ->selectRaw("{$haversine} AS distance", [$lat, $lng, $lat])
                ->with(['playground', 'user', 'teamPlayers.location'])
                ->withCount([
                    'teamPlayers',
                    'teamPlayers as blue_team_players_count' => function ($query) {
                        $query->where('team_side', 'blue');
                    },
                    'teamPlayers as red_team_players_count' => function ($query) {
                        $query->where('team_side', 'red');
                    }
                ])
                ->latest('matches.created_at')
                ->take(5);

            if ($catId) {
                $fallbackQuery->where('playgrounds.Category_ID', $catId);
            }

            $matches = $fallbackQuery->get();
        }

        return $matches;
    }
}
