<?php

namespace Modules\Match\Models;

use Carbon\Carbon;
use App\Traits\HasActive;
use Modules\User\Models\User;
use App\Enum\SubscriberStatus;
use Modules\Teams\Models\Team;
use App\Traits\HelperModelTrait;
use Illuminate\Database\Eloquent\Model;
use Modules\Playground\Models\Playground;
use Illuminate\Database\Eloquent\SoftDeletes;
use Modules\Challenges\Models\TeamMatchResult;
use Modules\Reviews\Models\Review;

class Matche extends Model
{
    use HelperModelTrait, SoftDeletes, HasActive;
    protected $table = 'matches';

    protected $fillable = [
        'playground_id',
        'organizer_user_id',
        'players_per_side',
        'name',
        'status',
        'gender',
        'price',
        'currency',
        'date',
        'About',
        'start_time',
        'end_time',
        'paid_online',
        'refund',
        'created_at',
        'updated_at',
        'deleted_at'
    ];

    protected $casts = [
        'paid_online' => 'boolean',
        'refund' => 'boolean',
        'status' => 'boolean',
    ];

    public function playground()
    {
        return $this->belongsTo(Playground::class);
    }


    public function user()
    {
        return $this->belongsTo(User::class, 'organizer_user_id');
    }

    public function organizer()
    {
        return $this->belongsTo(User::class, 'organizer_user_id');
    }

    public function teamPlayers()
    {
        return $this->belongsToMany(User::class, 'team_matches', 'match_id', 'user_id')->withPivot('team_side', 'created_at');
    }





    public function scopeInTime($query)
    {
        return $query->where(function ($q) {
            $q->where(fn($qu) => $qu->whereDate('date', '>', Carbon::now()->format('Y-m-d')));
            $q->orWhere(fn($qu) => $qu->whereDate('date', '=', Carbon::now()->format('Y-m-d'))->whereTime('start_time', '>', Carbon::now()->format('H:i:s')));
        })->active();
    }
    public function scopeCurrentTime($query)
    {
        return $query->where(function ($q) {
            $q->where(fn($qu) => $qu->whereDate('date', '>', Carbon::now()->format('Y-m-d')));
            $q->orWhere(fn($qu) => $qu->whereDate('date', '=', Carbon::now()->format('Y-m-d'))->whereTime('start_time', '<', Carbon::now()->format('H:i:s'))->whereTime('end_time', '>', Carbon::now()->format('H:i:s')));
        })->active();
    }
    public function scopeOutTime($query)
    {
        return $query->where(function ($q) {
            $q->where(fn($qu) => $qu->whereDate('date', '<', Carbon::now()->format('Y-m-d')));
            $q->orWhere(fn($qu) => $qu->whereDate('date', '=', Carbon::now()->format('Y-m-d'))->whereTime('start_time', '<', Carbon::now()->format('H:i:s'))->whereTime('end_time', '<', Carbon::now()->format('H:i:s')));
        })->active();
    }

    public function scopeActive($query)
    {
        return $query->where('matches.status', true);
    }
    public function scopeActiveSubscribersCount($query)
    {
        return $query->withCount(['subscribers' => fn($q) => $q->where('subscribers.status', SubscriberStatus::IN_PROGRESS)]);
    }

    public function scopeFilter($query)
    {
        return $query->when(request('area_id'), fn($q) => $q->whereHas('playground', fn($q) => $q->where('playgrounds.area_id', request('area_id'))))->when(request('playgrounds'), fn($q) => $q->whereIn('playground_id', request('playgrounds')))->when(request('date'), fn($q) => $q->whereIn('date', request('date')));
    }


    public function scopeNearest($query, $lat, $lng, $radius = 50)
    {
        if ($lat && $lng) {
            $haversine = "(6371 * acos(cos(radians(?)) 
                         * cos(radians(playgrounds.Latitude)) 
                         * cos(radians(playgrounds.Longitude) - radians(?)) 
                         + sin(radians(?)) 
                         * sin(radians(playgrounds.Latitude))))";

            return $query->select('matches.*')
                ->join('playgrounds', 'matches.playground_id', '=', 'playgrounds.id')
                ->selectRaw("{$haversine} AS distance", [$lat, $lng, $lat])
                ->whereRaw("{$haversine} <= ?", [$lat, $lng, $lat, $radius])
                ->orderBy('distance', 'asc');
        }
    }

    /**
     * @return bool
     */
    public function isPast()
    {
        if (!$this->date || !$this->end_time) {
            return false;
        }
        $endDateTime = \Carbon\Carbon::parse($this->date . ' ' . $this->end_time);
        return $endDateTime->lt(now());
    }


    public function setDateAttribute($value)
    {
        // أوتوماتيك أي تاريخ هيجي هيتحول للصيغة الصح
        $this->attributes['date'] = Carbon::parse($value)->format('Y-m-d');
    }


    public function reviews()
    {
        return $this->morphMany(Review::class, 'reviewable');
    }

    public function getStatusTextAttribute()
    {
        return $this->status ? __('active') : __('inactive');
    }
}
