<?php

namespace Modules\User\Models;

use Carbon\Carbon;
use App\Enum\UserType;
use App\Enum\NotifyType;
use App\Traits\HasActive;
use App\Traits\FullTextSearch;
use Modules\Areas\Models\Area;
use Modules\Teams\Models\Team;
use App\Traits\HelperModelTrait;
use Modules\Match\Models\Matche;
use App\Traits\DefaultMediaImage;
use Laravel\Sanctum\HasApiTokens;
use Spatie\MediaLibrary\HasMedia;
use Modules\Messages\Models\Message;
use Modules\Location\Models\Location;
use Illuminate\Notifications\Notifiable;
use Modules\Playground\Models\Playground;
use Spatie\MediaLibrary\InteractsWithMedia;
use Modules\Coaching\Models\CoachingRequest;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Modules\Payment\Models\Wallet;
use Modules\Reviews\Models\Review;

class User extends Authenticatable implements HasMedia
{
    use Notifiable,
        InteractsWithMedia,
        DefaultMediaImage,
        HasApiTokens,
        SoftDeletes,
        HasActive,
        HelperModelTrait;

    protected $searchable = ['FirstName', 'email', 'phone'];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'FirstName',
        'LastName',
        'password',
        'NickName',
        'Nationality',
        'phone',
        'email',
        'gender' ,
        'Date_Of_Birth',
        'longitude',
        'latitude',
        'position_id',
        'self_Ranking',
        'role_id',
        'balance',
        'currency',
        'status',
        'notification_settings'
    ];

    protected function casts(): array
    {
        return [
            'created_at' => 'datetime:Y-m-d h:i a',
            'password' => 'hashed',
            'status' => 'boolean',
         'Latitude' => 'float',
        'Longitude' => 'float',
        'notification_settings' => 'array',
        ];
    }
    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];


    public function location()
    {
        return $this->belongsTo(Location::class, 'position_id', 'id');
    }

    public function playgrounds()
    {
        return $this->hasMany(Playground::class);
    }


    public function setImageAttribute($image)
    {
        if ($image instanceof \Illuminate\Http\UploadedFile) {
            $this->clearMediaCollection('image');
            $this->addMedia($image)->toMediaCollection('image');
        }
    }
    public function getImageAttribute()
    {
        return $this->getFirstOrDefaultMediaUrl('image');
    }

    

    public function allNotifications()
    {
        return \Modules\Common\Models\Notification::where(function ($q) {
            $q->where(function ($query) {
                $query->whereDate('notifications.created_at', '>=', Carbon::parse($this->created_at)->format('Y-m-d'));
            });
        })->where(function ($q) {
            $q->where(function ($q) {
                $q->where('notifiable_id', $this->id)->where('notifiable_type', User::class);
            })->orWhere(function ($q) {
                $q->where('notifiable_id', 0)->where('notifiable_type', User::class)->wheredoesnthave('notificationActions', fn($q) => $q->where('user_id', $this->id)->where('type', NotifyType::DELETE));
            });
        })->with(['notificationActions' => fn($q) => $q->where('user_id', $this->id)]);
    }
    public function devices()
    {
        return $this->morphMany(Device::class, 'user', 'user_type', 'user_id')->latest();
    }

    public function scopeSearch($query)
    {
        return $query->when(request()->has('keyword'), function ($query, $word) {
            return $query->where('name', 'like', "%{$word}%")
                ->orWhere('email', 'like', "%{$word}%")->orWhere('mobile', 'like', "%{$word}%");
        });
    }
    public function matches()
    {
        return $this->belongsToMany(Matche::class, 'subscribers', 'user_id', 'matche_id')->withPivot('created_at', 'status');
    }

    public function teamMatches()
    {
        return $this->belongsToMany(Matche::class, 'team_matches', 'user_id', 'match_id')->withPivot('team_side', 'created_at');
    }
    public function matchesList()
    {
        return $this->hasMany(Matche::class, 'user_id', 'id');
    }


    public function role()
    {
        return $this->belongsTo(UserRole::class);
    }


    public function messages()
    {
        return $this->hasMany(Message::class, 'user_id');
    }

    public function getNameAttribute()
    {
        return $this->FirstName . ' ' . $this->LastName;
    }

    public function getFullPhoneAttribute()
    {
        return $this->phone;
    }


public function review()
    {
        return $this->hasMany(Review::class);
    }


    public function reviews()
    {
        return $this->morphMany(Review::class, 'reviewable');
    }


public function wallet() {
    return $this->hasOne(Wallet::class);
}




    public function wantsNotification($type)
    {
        if (empty($this->notification_settings)) {
            return true;
        }
        return $this->notification_settings[$type] ?? true;
    }




protected static function booted() {
    static::created(function ($user) {
        $user->wallet()->create(['balance' => 0]);
    });








}
}
