<?php

namespace Modules\User\Controllers\Admin;

use App\Enum\UserType;
use App\Helpers\ApiResponse;
use App\Notifications\BlockUserNotifications;
use App\Notifications\SupervisorNotifications;
use App\Notifications\UnSupervisorNotifications;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;
use Modules\Areas\Models\Area;
use Modules\Chats\Models\Chat;
use Modules\Common\Controllers\Admin\HelperController;
use Modules\Location\Models\Location;
use Modules\User\Models\Admin;
use Modules\User\Models\User;
use Modules\User\Requests\UserRequest;
use MshMsh\Helpers\ApiResponder;

class AdminController extends HelperController
{
    public function __construct()
    {
        $this->model = new User;
        $this->rows = User::when(request()->query('role_id'), fn($q) => $q->where('role_id', request()->query('role_id')))
            ->when(request()->has('status'), fn($q) => $q->where('status', request()->query('status')));

        $this->canAdd = false;
        if ((bool) request()->query('status')) {
            $this->canAdd = true;
        }

        $this->title = "Users";
        $this->name = 'users';
        $this->canShow = true;
        $this->formRequest = UserRequest::class;
        $this->roleName = 'User';
    }
    public function listBuilder()
    {
        $this->list = [
            'name' => __('name'),
            'mobile' => __('mobile'),
            'email' => __('email'),
        ];

        $this->links = [
            [
                'title' => 'Notify',
                'icon' => 'fa fa-bell',
                'url' => route('admin.notifications'),
                'key' => 'user_id',
                'type' => 'secondary',
            ]
        ];


        $this->switches['status'] = route('admin.users.active_status');
        $this->moreActions[] = 'syncActive';
    }
    public function formBuilder()
    {

        // $groups = get_select_data(UserGroup::get(), 'id', 'name');
        $locations = Location::active()->pluck('name', 'id')->toArray();
        $areas = Area::active()->pluck('name', 'id')->toArray();
        $this->inputs = [
            'name' => ['title' => __('name')],
            'mobile' => ['title' => __('phone')],
            'email' => ['title' => __('email')],
            'area_id' => ['title' => __('area'), 'type' => 'select', 'values' => $areas],
            'location_id' => ['title' => __('location'), 'type' => 'select', 'values' => $locations],
            'image' => ['title' => __('image'), 'type' => 'image', 'empty' => 1],
        ];
    }


    public function active_status(Request $request)
    {
        $user =  User::findOrFail($request->id) ;
        $user->update(['status' => !$user->status]);
        $user->refresh();
        // if (!$user->status) {
        //     Notification::send($user, new BlockUserNotifications($user));
        // }
        return ApiResponse::get('', ['status' => 1]);
    }
    public function active_supervisor(Request $request)
    {
        $user = User::findOrFail($request->id);
        if ($user->type == UserType::SUPERVISOR) {
            $user->update(['type' => UserType::CLIENT]);
            $user->notify(new UnSupervisorNotifications());
        } else {
            $user->update(['type' => UserType::SUPERVISOR]);
            $user->notify(new SupervisorNotifications());
        }
        return ApiResponse::get('', ['banned' => 1]);
    }

    public function syncActive($model)
    {
        if ($model->getRawOriginal('created_at') == $model->getRawOriginal('updated_at')) {
            $model->update(['status' => 1]);
        }
    }

    public function admin_login(Request $request)
    {
        if ($request->isMethod('GET')) {
            return view('User::auth.admin_login');
        }
        if (auth('admin')->attempt($request->only(['email', 'password']), true)) {
            if (auth('admin')->user()->role) {
                return redirect()->intended('/admin');
            }
        }
        return back()->with('error', __('Email or password is incorrect'));
    }
    public function show($id)
    {
        $user = User::findOrFail($id);
        $title =   __("show user");
        return view('User::admin.show', get_defined_vars());
    }
    public function notifications()
    {
        $notifications = auth('admin')->user()->notifications()->latest()->paginate(10);
        $title =   __("notifications");
        return view('User::admin.notifications', get_defined_vars());
    }

    public function admin_logout()
    {
        Auth::guard('admin')->logout();
        return redirect()->route('admin_login');
    }

    public function markNotifyRead()
    {
        $user = auth('admin')->user();
        $notifications = $user->unreadnotifications;
        foreach ($notifications as $notification) {
            $notification->markAsRead();
        }
    }

    public function saveToken()
    {
        $token = request()->token;
        $user = auth('admin')->user();
        $user->devices()->updateOrCreate(['token' => $token, 'platform' => request()->platform]);
        return response()->json(['status' => 'success']);
    }

    public function userSearch(Request $request)
    {
        $term = $request->get('term', '');
        $users = User::query()
            ->where(function ($q) use ($term) {
                $q->where('name', 'like', "%$term%")
                    ->orWhere('email', 'like', "%$term%")
                    ->orWhere('mobile', 'like', "%$term%")
                ;
            })
            ->limit(20)
            ->get(['id', 'name']);
        $results = $users->map(function ($user) {
            return [
                'id' => $user->id,
                'text' => $user->name,
            ];
        });
        return response()->json(['results' => $results]);
    }
}
