<?php

namespace Modules\Match\Services;

use App\Services\Services;
use Carbon\Carbon;
use Modules\Match\Models\Matche;

use Modules\Playground\Models\Playground;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\DB;

class MatchService extends Services
{

    private $colums = ['players_per_side', 'gender', 'status', 'price', 'date', 'start_time', 'end_time'];


    public function paginate()
    {
        $column = request('sort_by', 'id');
        $order = request('sort_order') ?: 'desc';
        $players_per_side = request('players_per_side', null);
        $price = request('price', null);
        $start_time = request('start_time', null);
        $end_time = request('end_time', null);
        $gender = request('gender', null);
        $startprice = request('startprice', null);
        $name = request('name', null);
        $endprice = request('endprice', null);
        $query = Matche::query()->with('user', 'playground')
            ->when(config('context.category_id'), function ($q, $catId) {
                $q->whereHas('playground', function ($q) use ($catId) {
                    $q->where('Category_ID', $catId);
                });
            })
            ->when($players_per_side, function ($q, $players_per_side) {
                $q->where('players_per_side', 'LIKE', "%$players_per_side%");
            })
            ->when($price, function ($q, $price) {
                $q->where('price', 'LIKE', "%$price%");
            })->when($name , function($q,$name){
                $q->where('name', 'LIKE', "%$name%");

            })
            ->when($start_time, function ($q, $start_time) {
                $q->where('start_time', 'LIKE', "%$start_time%");
            })
            ->when($gender, function ($q, $gender) {
                $q->where('gender', '=', $gender);
            })->when($startprice, function ($q, $startprice) {
                $q->where('price', '>=', $startprice);
            })
            ->when($endprice, function ($q, $endprice) {
                $q->where('price', '<=', $endprice);
            })
            ->when($column && array_key_exists($column, $this->colums), function ($q) use ($column, $order) {
                $q->orderBy($this->colums[$column], $order);
            });

        return $query->paginate(request('limit', 15));
    }

    public function store(array $data)
    {
        $playground = Playground::find($data['playground_id']);
        if (!$playground) {
            $this->setError(__('messages.Playground not found'));
            return false;
        }


      $hasError = false;


if($data['start_time'] && $data['end_time']){
    $start_time = Carbon::parse($data['start_time'])->format('H:i');
    $end_time = Carbon::parse($data['end_time'])->format('H:i');
    if ($start_time >= $end_time) {
        $this->setError(__('messages.Start time must be less than end time'));
        $hasError = true;
    }
}

        if ($playground && !empty($playground->slots)) {
            if (!in_array($data['players_per_side'], $playground->slots)) {
                $this->setError(__('messages.Invalid players per side for this playground'));
                $hasError = true;
            }
        }
        $date = Carbon::parse($data['date'])->format('Y-m-d');
        $newStart = $data['start_time'];
        $newEnd = $data['end_time'];

        $isBooked = Matche::where('playground_id', $data['playground_id'])
            ->where('date', $date)
            ->where(function ($query) use ($newStart, $newEnd) {
                $query->where('start_time', '<', $newEnd)
                    ->where('end_time', '>', $newStart)
                    ->where('status', '!=', 'cancelled');
            })->exists();

        if ($isBooked) {
        $this->setError(__('messages.Playground is booked at this time'));
        $hasError = true;
        }

        if($hasError){
            return false;
        }
        $data['organizer_user_id'] = auth('api')->id();
        $match = Matche::create($data);
        $match->status = 'open';
        $match->save();
        return $match;
    }


    public function update(array $data, $id)
    {
        $match = Matche::find($id);
        if (!$match) {
            $this->setError(__('messages.Match not found'));
            return false;
        }
        $match->update($data);
        return $match;
    }

    public function destroy($id)
    {
        $match = Matche::find($id);
        if (!$match) {
            $this->setError(__('messages.Match not found'));
            return false;
        }
        $match->delete();
        return $match;
    }

    public function show($id)
    {
        $match = Matche::with('user', 'playground', 'teamPlayers.location')
            ->withCount([
                'teamPlayers',
                'teamPlayers as white_team_players_count' => function ($query) {
                    $query->where('team_side', 'White');
                },
                'teamPlayers as dark_team_players_count' => function ($query) {
                    $query->where('team_side', 'Dark');
                }
            ])
            ->find($id);

        if (!$match) {
            $this->setError(__('messages.Match not found'));
            return false;
        }
        return $match;
    }

    public function upcommingmatches($playground_id)
    {
        $matches = Matche::where('playground_id', $playground_id)->where('status', '=', 'open')->where('date', '>=', now())->with('user', 'playground', 'teamPlayers.location')->withCount([
            'teamPlayers',
            'teamPlayers as white_team_players_count' => function ($query) {
                $query->where('team_side', 'White');
            },
            'teamPlayers as dark_team_players_count' => function ($query) {
                $query->where('team_side', 'Dark');
            }
        ])->paginate(request('limit', 15));

        return $matches;
    }

    public function joinMatch($matchId, $teamSide)
    {
        $user = auth('api')->user();

        if ($user->role->name != 'player') { 
            $this->setError(__('messages.You are not allowed to join this match'));
            return false;
        }


        if (!in_array($teamSide, ['White', 'Dark'])) {
            $this->setError(__('messages.Invalid team side'));
            return false;
        }

        return DB::transaction(function () use ($matchId, $teamSide, $user) {


            $match = Matche::where('id', $matchId)->lockForUpdate()->first();

            if (!$match) {
                $this->setError(__('messages.match_not_found'));
                return false;
            }

            $hasError = false;

            if ($match->date <= now()) {
                $this->setError(__('messages.match_is_closed_or_already_started'));
                $hasError = true;
            }

            // Check Status
            if ($match->status !== 'open') {
                $this->setError(__('messages.match_is_not_open_for_joining'));
                $hasError = true;
            }

            // 5. Gender Validation
            if ($match->gender !== $user->gender) {
                $this->setError(__('messages.Invalid gender'));
                $hasError = true;
            }


            if ($match->teamPlayers()->where('user_id', $user->id)->exists()) {
                $this->setError(__('messages.You have already joined this match'));
                $hasError = true;
            }

            // 7. Check capacity
            $currentCount = $match->teamPlayers()->wherePivot('team_side', $teamSide)->count();

            if ($currentCount >= $match->players_per_side) {
                $this->setError(__('messages.This team is full'));
                $hasError = true;
            }

            if ($hasError) {
                return false;
            }

            // 8. Attach User
            $match->teamPlayers()->attach($user->id, [
                'team_side' => $teamSide,
            ]);


            $owner = $match->organizer;
            if ($owner && $owner->id !== $user->id && $owner->wantsNotification('Players Joined My Match')) {

                $owner->notify(new \Modules\Match\Notifications\PlayerJoinedMatchNotification($match, $user));
            }

            return $match;
        });
    }



    public function getNearest($lat, $lng)
    {
        $catId = config('context.category_id');

        $query = Matche::active()
            ->with(['playground', 'user', 'teamPlayers.location'])
            ->withCount([
                'teamPlayers',
                'teamPlayers as blue_team_players_count' => function ($query) {
                    $query->where('team_side', 'blue');
                },
                'teamPlayers as red_team_players_count' => function ($query) {
                    $query->where('team_side', 'red');
                }
            ]);

        if ($catId) {
            $query->whereHas('playground', function ($q) use ($catId) {
                $q->where('Category_ID', $catId);
            });
        }

        $matches = $query->nearest($lat, $lng)
            ->take(5)
            ->get();

        if ($matches->isEmpty()) {
            // Fallback: Get latest matches but still calculate distance for display
            $haversine = "(6371 * acos(cos(radians(?)) 
                         * cos(radians(playgrounds.Latitude)) 
                         * cos(radians(playgrounds.Longitude) - radians(?)) 
                         + sin(radians(?)) 
                         * sin(radians(playgrounds.Latitude))))";

            $fallbackQuery = Matche::active()
                ->select('matches.*')
                ->join('playgrounds', 'matches.playground_id', '=', 'playgrounds.id')
                ->selectRaw("{$haversine} AS distance", [$lat, $lng, $lat])
                ->with(['playground', 'user', 'teamPlayers.location'])
                ->withCount([
                    'teamPlayers',
                    'teamPlayers as blue_team_players_count' => function ($query) {
                        $query->where('team_side', 'blue');
                    },
                    'teamPlayers as red_team_players_count' => function ($query) {
                        $query->where('team_side', 'red');
                    }
                ])
                ->latest('matches.created_at')
                ->take(5);

            if ($catId) {
                $fallbackQuery->where('playgrounds.Category_ID', $catId);
            }

            $matches = $fallbackQuery->get();
        }

        return $matches;
    }
}
