<?php

namespace Modules\Playground\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Modules\Playground\Models\Playground;
use Modules\User\Models\Admin;
use Modules\User\Models\User;
use Illuminate\Support\Facades\Notification;
use App\Notifications\AdminNotifications;
use App\Notifications\NewNearbyPlaygroundNotification;

class ProcessPlaygroundNotifications implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $playground;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(Playground $playground)
    {
        $this->playground = $playground;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    /**
     * The number of times the job may be attempted.
     *
     * @var int
     */
    public $tries = 3;

    /**
     * The number of seconds the job can run before timing out.
     *
     * @var int
     */
    public $timeout = 120;

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        $admins = Admin::active()->get();
        Notification::send($admins, new AdminNotifications(
            'New Playground Created',
            "A new playground '{$this->playground->Name}' has been created.",
            [
                'playground_id' => $this->playground->id,
                'action' => 'playground_approval',
                'link' => route('admin.playgrounds.edit', $this->playground->id)
            ]
        ));

        //  Notify nearby users (within 10km)
        //  Using chunking to handle large number of users efficiently
        $radius = 10;
        User::select('users.*')
            ->selectRaw("( 6371 * acos( cos( radians(?) ) * cos( radians( latitude ) ) * cos( radians( longitude ) - radians(?) ) + sin( radians(?) ) * sin( radians( latitude ) ) ) ) AS distance", [$this->playground->Latitude, $this->playground->Longitude, $this->playground->Latitude])
            ->having('distance', '<', $radius)
            ->chunk(100, function ($users) {
                Notification::send($users, new NewNearbyPlaygroundNotification($this->playground));
            });
    }
}
