<?php

namespace Modules\Match\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Cache; 
use Illuminate\Support\Facades\Log;
use App\Notifications\MatchNotifications;
use Modules\Match\Models\Matche;

class SendMatchReminderJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $match;
    public $minutes;
    public $tries = 3;

    public function __construct(Matche $match, int $minutes)
    {
        $this->match = $match;
        $this->minutes = $minutes;
    }

    public function handle()
    {
        
        $cacheKey = "match_reminder_{$this->match->id}_{$this->minutes}";
        if (Cache::has($cacheKey)) {
            return;
        }

        Log::info("SendMatchReminderJob: Started for Match {$this->match->id}");

        $this->match->load(['teamPlayers.devices', 'playground']);

   
        $players = $this->match->teamPlayers->filter(function ($user) {
           
            if (!$user->wantsNotification('Match Start Reminders')) return false;
            
            if ($user->devices->isEmpty()) return false;

            return true;
        });

        if ($players->isEmpty()) {
            Log::info("SendMatchReminderJob: No valid players found for Match {$this->match->id}");
      
            Cache::put($cacheKey, true, now()->addMinutes(60));
            return;
        }

        $playgroundName = $this->match->playground?->name ?? 'Playground';
        $message = "Reminder: Match at {$playgroundName} will start in {$this->minutes} minutes.";

        Log::info("SendMatchReminderJob: Sending to " . $players->count() . " players.");
        
      
        Notification::send($players, new MatchNotifications($this->match, $message));

        Cache::put($cacheKey, true, now()->addMinutes(60));
    }
}