<?php

namespace App\Support\Actions;

use Illuminate\Database\Eloquent\Model;
use Kreait\Firebase\Contract\Messaging;
use Kreait\Firebase\Messaging\AndroidConfig;
use Kreait\Firebase\Messaging\ApnsConfig;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;

class FCMAction
{
    public $firebaseTokens;
    public $firebaseToken;
    public $messageType;
    public $topic;
    public $messaging;
    public $data;
    public $title;
    public $body;
    public $imageUrl;
    public $clickAction;

    public function __construct($userOrUsers = null)
    {
        \Illuminate\Support\Facades\Log::info("FCMAction: Constructor called. Input type: " . gettype($userOrUsers));
        if (is_object($userOrUsers)) {
            \Illuminate\Support\Facades\Log::info("FCMAction: Input class: " . get_class($userOrUsers));
        }

        $this->messaging = app('firebase.messaging');
        $this->messageType = 'default';
        $this->topic = 'topic';
        $this->data = ['fcmData' => 'Start data'];
        $this->title = '';
        $this->body = '';
        $this->imageUrl = '';
        $this->clickAction = '';
        if (is_array($userOrUsers)) {
            // Array of users
            $this->firebaseTokens = collect($userOrUsers)
                ->flatMap(function ($user) {
                    return isset($user->devices) ? $user->devices->pluck('token') : [];
                })
                ->filter()
                ->unique()
                ->values()
                ->toArray();
            $this->firebaseToken = null;
        } elseif ($userOrUsers instanceof Model) {
            // Single user
            if (isset($userOrUsers) && method_exists($userOrUsers, 'devices')) {
                \Illuminate\Support\Facades\Log::info("FCMAction: Fetching device for User ID: " . $userOrUsers->id);
                $device = $userOrUsers->devices()->orderBy("id", 'desc')->first();
                if ($device) {
                    \Illuminate\Support\Facades\Log::info("FCMAction: Found device ID: {$device->id} with token: " . substr($device->token, 0, 10) . "...");
                    $this->firebaseToken = $device->token;
                } else {
                    \Illuminate\Support\Facades\Log::info("FCMAction: No device found for User ID: " . $userOrUsers->id);
                    $this->firebaseToken = "";
                }
            } else {
                $this->firebaseToken = "";
            }
            $this->firebaseTokens = [];
        }
    }

    public static function new($userOrUsers)
    {
        return new self($userOrUsers);
    }

    public function withData(array $array)
    {
        foreach ($array as $key => $value) {
            if (is_string($value)) {
                $this->data[$key] = $value;
            }
            if (is_array($value)) {
                $this->withData($value);
            }
        }
        return $this;
    }

    public function withMessageType($messageType)
    {
        $this->messageType = $messageType;
        return $this;
    }

    public function withTopic($topic)
    {
        $this->topic = $topic;
        return $this;
    }

    public function withClickAction($clickAction)
    {
        $this->clickAction = $clickAction;
        return $this;
    }

    public function withImageUrl($imageUrl)
    {
        $this->imageUrl = $imageUrl;
        return $this;
    }

    public function withTitle($title)
    {
        $this->title = $title;
        return $this;
    }

    public function withBody($body)
    {
        $this->body = $body;
        return $this;
    }

    public function sendMessage($type = 'topic')
    {
        \Illuminate\Support\Facades\Log::info('FCMAction: sendMessage called with type: ' . $type);
        $message = $this->createMessage()->withHighestPossiblePriority()->withApnsConfig(ApnsConfig::fromArray([
            'payload' => [
                'aps' => [
                    'sound' => 'note.mp3',
                ],
            ],
            'fcm_options' => [],
            'headers' => [],
        ]))->withAndroidConfig(
            AndroidConfig::fromArray([
                'notification' => [
                    'sound' => 'note.mp3',
                ],
            ])
        );
        if ($type === 'tokens') {
            if (!empty($this->firebaseTokens)) {
                \Illuminate\Support\Facades\Log::info('FCMAction: Sending multicast to ' . count($this->firebaseTokens) . ' tokens.');
                $this->messaging->sendMulticast($message, $this->firebaseTokens);
            } elseif (!empty($this->firebaseToken)) {
                \Illuminate\Support\Facades\Log::info('FCMAction: Sending multicast to single token: ' . $this->firebaseToken);
                $this->messaging->sendMulticast($message, [$this->firebaseToken]);
            } else {
                \Illuminate\Support\Facades\Log::info('FCMAction: No tokens found to send message.');
            }
        }
        if ($type === 'topic') {
            \Illuminate\Support\Facades\Log::info('FCMAction: Sending to topic: ' . $this->topic);
            $this->messaging->send($message);
        }
    }

    private function createMessage()
    {
        return CloudMessage::withTarget('topic', $this->topic)
            ->withNotification(Notification::create($this->title, $this->body, $this->imageUrl))
            ->withData($this->data);
    }
}
