<?php

namespace Modules\Playground\Models;

use App\Traits\HasActive;
use App\Traits\HelperModelTrait;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Modules\Areas\Models\Area;
use Modules\Categories\Models\Category;
use Modules\User\Models\User;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use App\Traits\DefaultMediaImage;
use App\Traits\HasUnifiedMedia;
use App\Traits\Translatable;
use Laravel\Scout\Searchable;
use Modules\Match\Models\Matche;
use Modules\Reviews\Models\Review;

class Playground extends Model implements HasMedia
{
    use HasFactory, SoftDeletes, HasActive, HelperModelTrait, HasUnifiedMedia, DefaultMediaImage, Searchable , Translatable;

    protected $searchable = ['Name', 'Address', 'City', 'Telephone', 'Email'];

    protected $translatable = ['Name', 'Owner', 'About', 'Address', 'City', 'Grass_Type', 'Accessibility'];



    protected $fillable = [
        'Site_ID',
        'Name',
        'Name_ar',
        'Owner',
        'Owner_ar',
        'Owner_ID',
        'Address',
        'Address_ar',
        'City',
        'City_ar',
        'Telephone',
        'Email',
        'Grass_Type',
        'Grass_Type_ar',
        'Accessibility',
        'Accessibility_ar',
        'status',
        'Disability_Parking',
        'Disability_Activity_Areas',
        'Disability_Social_Areas',
        'Disability_Spectator_Areas',
        'Disability_Changing_Facilities',
        'Disability_Toilets',
        'Disability_Finding_and_Reaching_the_Entrance',
        'Disability_Reception_Area',
        'Disability_Doorways',
        'Disability_Emergency_Exits',
        'Latitude',
        'Longitude',
        "About",
        "About_ar",
        'Category_ID',
        'media',
        'location',
        'slots',
    ];



    public function toSearchableArray()
    {
        return [
            'id' => $this->id,
            'Name' => $this->Name, // البحث بالاسم
            'Address' => $this->Address, // البحث بالعنوان
            'City' => $this->City, // البحث بالعنوان

        ];
    }


    public function getLocationAttribute()
    {

        return (object)[
            'latitude' => $this->Latitude,
            'longitude' => $this->Longitude,
            'address' => $this->Address
        ];
    }


    public function setLocationAttribute($value)
    {
        if (is_array($value)) {
            $this->attributes['Latitude'] = $value['lat'] ?? ($value['latitude'] ?? null);
            $this->attributes['Longitude'] = $value['lng'] ?? ($value['longitude'] ?? null);
            $this->attributes['Address'] = $value['text'] ?? ($value['address'] ?? null);
        }
    }


    protected $casts = [
        'Disability_Parking' => 'boolean',
        'Disability_Activity_Areas' => 'boolean',
        'Disability_Social_Areas' => 'boolean',
        'Disability_Spectator_Areas' => 'boolean',
        'Disability_Changing_Facilities' => 'boolean',
        'Disability_Toilets' => 'boolean',
        'Disability_Finding_and_Reaching_the_Entrance' => 'boolean',
        'Disability_Reception_Area' => 'boolean',
        'Disability_Doorways' => 'boolean',
        'Disability_Emergency_Exits' => 'boolean',
        'slots' => 'array',
        'Latitude' => 'float',
        'Longitude' => 'float',
    ];

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function getStatusHtmlAttribute()
    {
        $route = route('admin.playgrounds.active_status');
        $options = [
            'active' => __('Active'),
            'inactive' => __('Inactive'),
            'pending' => __('pending')
        ];
        $html = '<select class="form-control status-select" style="min-width: 100px;" data-id="' . $this->id . '" data-route="' . $route . '">';
        foreach ($options as $key => $label) {
            $selected = $this->status == $key ? 'selected' : '';
            $html .= '<option value="' . $key . '" ' . $selected . '>' . $label . '</option>';
        }
        $html .= '</select>';
        return $html;
    }


    public function scopeWithDistance($query, $lat, $lng)
    {
        if ($lat && $lng) {
            $distanceQuery = "
            ( 6371 * acos( cos( radians(?) ) *
            cos( radians( Latitude ) )
            * cos( radians( Longitude ) - radians(?) )
            + sin( radians(?) ) *
            sin( radians( Latitude ) ) ) )
            AS distance
            ";
            return $query->select('playgrounds.*')
                ->selectRaw($distanceQuery, [$lat, $lng, $lat]);
        }
    }

    public function scopeNearest($query, $lat, $lng, $radius = 50)
    {
        if ($lat && $lng) {
            return $query->withDistance($lat, $lng)
                ->having('distance', '<=', $radius)
                ->orderBy('distance', 'asc');
        }
    }


    public function getCategoryNameAttribute()
    {
        return @$this->category->name;
    }

    public function category()
    {
        return $this->belongsTo(Category::class, 'Category_ID');
    }



    public function user()
    {
        return $this->belongsTo(User::class, 'Owner_ID');
    }


    public function match()
    {
        return $this->hasMany(Matche::class);
    }



    public function reviews()
    {
        return $this->morphMany(Review::class, 'reviewable');
    }
}
