<?php

namespace Modules\Match\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Modules\Match\Models\Matche;
use Modules\User\Models\User;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Cache;
use App\Notifications\MatchesMissingPlayersNotification;
use Illuminate\Support\Facades\Log;

class NotifyMissingPlayersJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $tries = 3;
    public $timeout = 120;

    protected $match;

    public function __construct(Matche $match)
    {
        $this->match = $match;
    }

    public function handle()
    {

        $this->match->load(['playground', 'teamPlayers:id']);

        $playground = $this->match->playground;

        if (!$playground || !$playground->Latitude || !$playground->Longitude) {
            return;
        }


        $maxPlayers = $this->match->players_per_side * 2;
        if ($this->match->teamPlayers->count() >= $maxPlayers || $this->match->status !== 'open') {
            return;
        }


        $existingPlayerIds = $this->match->teamPlayers->pluck('id')->toArray();

        $lat = $playground->Latitude;
        $lng = $playground->Longitude;
        $radius = 20;


        User::select('id', 'FirstName', 'latitude', 'longitude')
            ->selectRaw("( 6371 * acos( cos( radians(?) ) * cos( radians( latitude ) ) * cos( radians( longitude ) - radians(?) ) + sin( radians(?) ) * sin( radians( latitude ) ) ) ) AS distance", [$lat, $lng, $lat])
            ->having('distance', '<=', $radius)
            ->whereNotIn('id', $existingPlayerIds)
            ->chunk(100, function ($users) {
                $usersToNotify = $users->filter(function ($user) {
                    // 1. Check Notification Settings
                    if (!$user->wantsNotification('Matches Missing Players')) {
                        return false;
                    }

                    // 2. Check Cache (Prevent Spam)
                    $cacheKey = "notified_missing_match_{$this->match->id}_user_{$user->id}";
                    if (Cache::has($cacheKey)) {
                        return false; // Already notified
                    }

                    return true;
                });
Log::info("Found " . $usersToNotify->count() . " users to notify for match " . $this->match->id);
                if ($usersToNotify->isNotEmpty()) {
                    Notification::send($usersToNotify, new MatchesMissingPlayersNotification($this->match));

                    // 3. Set Cache for notified users
                    foreach ($usersToNotify as $user) {
                        $cacheKey = "notified_missing_match_{$this->match->id}_user_{$user->id}";
                      Log::info("Notified user {$user->id} about missing players for match {$this->match->id}");
                        // Cache for 24 hours
                        Cache::put($cacheKey, true, now()->addHours(24));
                    }
                }
            });
    }
}
