<?php

namespace Modules\User\Models;

use Carbon\Carbon;
use App\Enum\UserType;
use App\Enum\NotifyType;
use App\Traits\HasActive;
use App\Traits\FullTextSearch;
use Modules\Areas\Models\Area;
use Modules\Teams\Models\Team;
use App\Traits\HelperModelTrait;
use Modules\Match\Models\Matche;
use App\Traits\DefaultMediaImage;
use Laravel\Sanctum\HasApiTokens;
use Spatie\MediaLibrary\HasMedia;
use Modules\Messages\Models\Message;
use Modules\Location\Models\Location;
use Illuminate\Notifications\Notifiable;
use Modules\Playground\Models\Playground;
use Spatie\MediaLibrary\InteractsWithMedia;
use Modules\Coaching\Models\CoachingRequest;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;

class User extends Authenticatable implements HasMedia
{
    use Notifiable,
        InteractsWithMedia,
        DefaultMediaImage,
        HasApiTokens,
        SoftDeletes,
        HasActive,
        HelperModelTrait;

    protected $searchable = ['name', 'email', 'mobile'];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'email',
        'status',
        'mobile',
        'new_mobile',
        'password',
        'lang',
        'banned',
        'area_id',
        'location_id',
        'type',
        'notify',
        'image',
        'missed_match_count',
        'banned_from',
        'banned_to',
    ];

    protected function casts(): array
    {
        return [
            'created_at' => 'datetime:Y-m-d h:i a',
            'password' => 'hashed',
            'banned' => 'boolean',
            'notify' => 'boolean',
            'status' => 'boolean',
        ];
    }
    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    public function area()
    {
        return $this->belongsTo(Area::class);
    }
    public function location()
    {
        return $this->belongsTo(Location::class);
    }


    public function setPasswordAttribute($password)
    {
        $this->attributes['password'] = bcrypt($password);
    }

    public function otps()
    {
        return $this->hasMany(Otp::class);
    }
    public function getMycodeAttribute()
    {
        return @Otp::where('user_id', $this->id)->where('status', false)->whereDate('expired_at', '>', Carbon::now()->format('Y-m-d'))->first()->otp;
    }
    public function scopeOtpCode($value)
    {
        return $value->otps()->where('status', false)->whereDate('expired_at', '>', Carbon::now()->format('Y-m-d'));
    }
    public function setImageAttribute($image)
    {
        if (is_uploaded_file($image)) {
            $this->clearMediaCollection('image');
            $this->addMediaFromRequest('image')
                ->toMediaCollection('image');
        }
    }
    public function getImageAttribute()
    {
        return $this->getFirstOrDefaultMediaUrl('image');
    }
    public function getFullPhoneAttribute()
    {
        return '+966' . ltrim(@$this->attributes['mobile'], '0');
    }

    public function playgrounds()
    {
        return $this->belongsToMany(Playground::class, 'user_playgrounds', 'user_id', 'playground_id');
    }
    public function allNotifications()
    {
        return \Modules\Common\Models\Notification::where(function ($q) {
            $q->where(function ($query) {
                $query->whereDate('notifications.created_at', '>=', Carbon::parse($this->created_at)->format('Y-m-d'));
            });
        })->where(function ($q) {
            $q->where(function ($q) {
                $q->where('notifiable_id', $this->id)->where('notifiable_type', User::class);
            })->orWhere(function ($q) {
                $q->where('notifiable_id', 0)->where('notifiable_type', User::class)->wheredoesnthave('notificationActions', fn($q) => $q->where('user_id', $this->id)->where('type', NotifyType::DELETE));
            });
        })->with(['notificationActions' => fn($q) => $q->where('user_id', $this->id)]);
    }
    public function devices()
    {
        return $this->morphMany(Device::class, 'user', 'user_type', 'user_id')->latest();
    }

    public function scopeSearch($query)
    {
        return $query->when(request()->has('keyword'), function ($query, $word) {
            return $query->where('name', 'like', "%{$word}%")
                ->orWhere('email', 'like', "%{$word}%")->orWhere('mobile', 'like', "%{$word}%");
        });
    }
    public function matches()
    {
        return $this->belongsToMany(Matche::class, 'subscribers', 'user_id', 'matche_id')->withPivot('created_at', 'status');
    }
    public function matchesList()
    {
        return $this->hasMany(Matche::class, 'user_id', 'id');
    }
    public function supervisor_matches()
    {
        return $this->hasMany(Matche::class, 'supervisor_id', 'id');
    }

    public function getIssupervisorAttribute()
    {
        return $this->type == UserType::SUPERVISOR;
    }
    public function coachingRequest()
    {
        return $this->hasOne(CoachingRequest::class, 'user_id');
    }

    /**
     * Get all teams this user belongs to
     */
    public function teams()
    {
        return $this->belongsToMany(Team::class, 'team_user')
            ->withTimestamps();
    }

    /**
     * Get teams where this user is a leader
     */
    public function leaderTeams()
    {
        return $this->belongsToMany(Team::class, 'team_user')
            ->where('team_user.role', 'leader')
            ->withTimestamps();
    }

    /**
     * Get teams where this user is a subleader
     */
    public function subLeaderTeams()
    {
        return $this->belongsToMany(Team::class, 'team_user')
            ->where('team_user.role', 'subLeader')
            ->withTimestamps();
    }

    /**
     * Get teams where this user is a member
     */
    public function memberTeams()
    {
        return $this->belongsToMany(Team::class, 'team_user')
            ->where('team_user.role', 'member')
            ->withTimestamps();
    }

    /**
     * Check if user is leader of any team
     */
    public function isTeamLeader()
    {
        return $this->leaderTeams()->exists();
    }

    /**
     * Check if user is leader of specific team
     */
    public function isLeaderOf(Team $team)
    {
        return $this->leaderTeams()->where('team_id', $team->id)->exists();
    }

    public function messages()
    {
        return $this->hasMany(Message::class, 'user_id');
    }
}
