<?php

namespace Modules\User\Controllers\Api;

use App\Enum\UserType;
use App\Helpers\ApiResponse;
use App\Http\Controllers\Controller;
use App\Notifications\Couching\AdminNotifications;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;
use Illuminate\Validation\ValidationException;
use Modules\Coaching\Models\CoachingRequest;
use Modules\Match\Requests\Subscribe;
use Modules\User\Models\Admin;
use Modules\User\Models\Otp;
use Modules\User\Requests\ConfirmMobileRequest;
use Modules\User\Requests\ProfileUpdateRequest;
use Modules\User\Resources\UserResource;

class ApiController extends Controller
{
    public function profile()
    {
        $user = auth('api')->user()->load('location','area');
        return ApiResponse::loaded([
            'user' => new UserResource($user)
        ]);
    }
    public function update(ProfileUpdateRequest $request)
    {
        $user = auth('api')->user();
        $data = $request->validated();
        $user->update($data);
        $user->refresh();
        if($user->new_mobile){
            $code = Confirmation::sendCode($user);
        }
        return ApiResponse::updated([
            'user' => new UserResource($user),
            'mobile_changed' => (bool) $user->new_mobile,
            'code' => (int) @$code,
        ]);
    }
    public function confirm_mobile(ConfirmMobileRequest $request)
    {
        $user = auth('api')->user();
        $token = Otp::where('otp', $request->code)->where('status',false)
            ->whereRelation('user', 'new_mobile', '=', $user->new_mobile)
            ->first();
        if (!$token) {
            return ApiResponse::failed(__('Activation code is not correct'));
        }
        if ($token->expired_at < Carbon::now()->format('Y-m-d H:i:s')) {
            return ApiResponse::failed(__('Activation code is expired'));
        }
        $user->update(['mobile' => $user->new_mobile, 'new_mobile' => null]);
        $access_token = $user->createToken('authToken')->plainTextToken;
        return ApiResponse::loaded([
            'user' => UserResource::make($user),
            'access_token' => $access_token,
        ],200,__('messages.Your mobile changed successfully'));
    }
    public function delete()
    {
        $user = auth('api')->user();
        $user->delete();
        return ApiResponse::deleted(200,__('messages.Your account deleted successfully'));
    }

    public function coachRequest(Request $request)
    {
        $user = auth('api')->user();
        throw_if($user->type == UserType::COACH, ValidationException::withMessages(['msg' => __('messages.you already a coach')]));
        $existCoach = CoachingRequest::where('user_id', $user->id)->first();
        throw_if(@$existCoach->status == \App\Enum\CoachingRequest::PENDING, ValidationException::withMessages(['msg' => __('messages.you already sent a request for to be a couch')]));
        $coach = CoachingRequest::firstOrCreate([
            'user_id' => $user->id
        ]);
        throw_if($coach->status == \App\Enum\CoachingRequest::REFUSED, ValidationException::withMessages(['msg' => __('messages.your request to be a coach is refused')]));
        $admins = Admin::active()->get();
        foreach ($admins as $admin){
            Notification::send($admin,new AdminNotifications($coach));
        }
        return ApiResponse::created();
    }
}
