<?php

namespace Modules\Teams\Models;

use App\Traits\HasActive;
use Modules\User\Models\User;
use Modules\Areas\Models\Area;
use App\Traits\HelperModelTrait;
use Modules\Match\Models\Matche;
use Modules\Messages\Models\Message;
use Modules\Teams\Models\TeamRanking;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Modules\Challenges\Models\TeamMatchResult;
use Illuminate\Database\Eloquent\Factories\HasFactory;


class Team extends Model
{
    use HasFactory, HasActive, SoftDeletes, HelperModelTrait;

    protected $fillable = [
        'name',
        'status',
        'logo',
        'area_id',
        'bio',
        'remuntada_challenge_league',
        'remuntada_elite_cup',
        'remuntada_super_cup',
    ];

    /**
     * Get all users that belong to this team
     */
    public function users()
    {
        return $this->belongsToMany(User::class, 'team_user')
            ->withTimestamps()
            ->withPivot('role');
    }

    public function area()
    {
        return $this->belongsTo(Area::class);
    }

    /**
     * Get the team leader
     */
    public function leader()
    {
        return $this->belongsToMany(User::class, 'team_user')
            ->where('team_user.role', 'leader')
            ->withTimestamps();
    }

    /**
     * Get team sub-leaders
     */
    public function subLeaders()
    {
        return $this->belongsToMany(User::class, 'team_user')
            ->where('team_user.role', 'subLeader')
            ->withTimestamps();
    }

    /**
     * Get team members (excluding leaders)
     */
    public function members()
    {
        return $this->belongsToMany(User::class, 'team_user')
            ->where('team_user.role', 'member')
            ->withTimestamps();
    }

    /**
     * Check if user is leader of this team
     */
    public function isLeader(User $user)
    {
        return $this->leader()->where('user_id', $user->id)->exists();
    }

    /**
     * Check if user is subleader of this team
     */
    public function isSubLeader(User $user)
    {
        return $this->subLeaders()->where('user_id', $user->id)->exists();
    }

    /**
     * Get all matches where this team is team1
     */
    public function matchesAsTeam1()
    {
        return $this->hasMany(Matche::class, 'team1_id');
    }

    /**
     * Get all matches where this team is team2
     */
    public function matchesAsTeam2()
    {
        return $this->hasMany(Matche::class, 'team2_id');
    }

    public function matchResults()
    {
        return $this->hasMany(TeamMatchResult::class, 'team_id');
    }

    public function ranking()
    {
        return $this->hasOne(TeamRanking::class);
    }

    /**
     * Get competitive ranking for this team
     */
    public function competitiveRanking()
    {
        return $this->hasOne(TeamRanking::class)->where('is_competitive', true);
    }

    /**
     * Get friendly ranking for this team
     */
    public function friendlyRanking()
    {
        return $this->hasOne(TeamRanking::class)->where('is_competitive', false);
    }

    /**
     * Get all rankings for this team
     */
    public function rankings()
    {
        return $this->hasMany(TeamRanking::class);
    }

    public function messages()
    {
        return $this->hasMany(Message::class, 'team_id');
    }
}
