<?php

namespace Modules\Teams\Controllers;

use Illuminate\Support\Str;
use App\Helpers\ApiResponse;
use Modules\User\Models\User;
use Modules\Teams\Models\Team;
use Modules\Teams\Models\TeamUser;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;
use Modules\Teams\Requests\TeamRequest;
use Modules\Teams\Requests\MemberRequest;
use Modules\Teams\Resources\TeamResource;
use Modules\Teams\Requests\ChangeRoleRequest;
use Modules\Teams\Requests\UpdateTeamRequest;
use Modules\Common\Models\Notification;
use Illuminate\Notifications\Notification as BaseNotification;
use App\Notifications\UserAddedToTeamNotification;
use App\Notifications\UserRemovedFromTeamNotification;
use App\Notifications\UserRoleChangedNotification;

class ApiController extends Controller
{
    public function allActiveTeams()
    {
        $teams = Team::where('status', true)
            ->with(['competitiveRanking', 'friendlyRanking'])
            ->paginate(10);
        return ApiResponse::loaded(TeamResource::collection($teams));
    }

    public function show($id)
    {
        $team = Team::with(['users', 'competitiveRanking', 'friendlyRanking'])->findOrFail($id);
        return ApiResponse::loaded(new TeamResource($team));
    }

    public function store(TeamRequest $request)
    {
        $data = $request->validated();

        if ($request->hasFile('logo')) {
            $uuid = (string) Str::uuid();
            $file = $request->file('logo');
            $ext = $file->getClientOriginalExtension();
            $filename = "{$uuid}.{$ext}";

            $path = $file->storeAs('teams', $filename, 'public');

            $data['logo'] = $path;
        }

        $team = Team::create($data);

        $userId = auth('api')->user()->id;

        $team->users()->attach($userId, ['role' => 'leader']);

        return ApiResponse::created(new TeamResource($team->fresh()));
    }

    public function update(UpdateTeamRequest $request, $id)
    {
        $team = Team::findOrFail($id);
        $user = auth('api')->user();
        if (!$team->isLeader($user)) {
            return ApiResponse::failed(__("teams.only_leader_can_update"), 403);
        }
        $data = $request->validated();

        if ($request->hasFile('logo')) {

            if ($team->logo) {
                Storage::disk('public')->delete($team->logo);
            }
            $uuid = (string) Str::uuid();
            $file = $request->file('logo');
            $ext = $file->getClientOriginalExtension();
            $filename = "{$uuid}.{$ext}";
            $path = $file->storeAs('teams', $filename, 'public');
            $data['logo'] = $path;
        }

        $team->update($data);

        return ApiResponse::updated(new TeamResource($team->fresh()));
    }

    public function destroy($id)
    {
        $team = Team::findOrFail($id);
        $user = auth('api')->user();
        if (!$team->isLeader($user)) {
            return ApiResponse::failed(__("teams.only_leader_can_delete"), 403);
        }

        if ($team->logo) {
            Storage::disk('public')->delete($team->logo);
        }
        $team->users()->detach();
        $team->delete();
        return ApiResponse::deleted();
    }

    public function userTeams()
    {
        $user = auth('api')->user()->load(['teams' => function ($query) {
            $query->with(['competitiveRanking', 'friendlyRanking']);
        }]);
        return ApiResponse::loaded(TeamResource::collection($user->teams));
    }

    public function addMember(MemberRequest $request)
    {
        $data = $request->validated();

        $team = Team::findOrFail($data['team_id']);
        $currentUser = auth('api')->user();

        // Only leaders or subleaders can add members
        if (!($team->isLeader($currentUser) || $team->isSubLeader($currentUser))) {
            return ApiResponse::failed(__("teams.only_leader_or_subleader_can_add"), 403);
        }

        // Find user by phone number
        $userToAdd = User::where('mobile', $data['phone_number'])->first();
        if (!$userToAdd) {
            return ApiResponse::failed(__("teams.phone_not_associated"), 400);
        }

        // Prevent adding themselves
        if ($currentUser->id == $userToAdd->id) {
            return ApiResponse::failed(__("teams.cannot_add_self"), 403);
        }

        // Check if user is already a member
        if (TeamUser::where(['team_id' => $team->id, 'user_id' => $userToAdd->id])->exists()) {
            return ApiResponse::failed(__("teams.already_member"), 400);
        }

        $team->users()->attach($userToAdd->id, ['role' => 'member']);
        // Send notification to the user
        $userToAdd->notify(new UserAddedToTeamNotification($team, $currentUser));
        return ApiResponse::success(__("teams.user_added_success"));
    }

    public function removeMember(MemberRequest $request)
    {
        $data = $request->validated();

        $team = Team::findOrFail($data['team_id']);
        $currentUser = auth('api')->user();

        // Find user to remove by phone number
        $userToRemove = User::where('mobile', $data['phone_number'])->first();
        if (!$userToRemove) {
            return ApiResponse::failed(__("teams.phone_not_associated"), 400);
        }

        // Check if user to remove is a member of the team
        $teamUserToRemove = $team->users()->where('user_id', $userToRemove->id)->first();
        if (!$teamUserToRemove) {
            return ApiResponse::failed(__("teams.not_member"), 400);
        }

        // Get roles
        $currentUserRole = TeamUser::where('team_id', $team->id)->where('user_id', $currentUser->id)->value('role');
        $userToRemoveRole = TeamUser::where('team_id', $team->id)->where('user_id', $userToRemove->id)->value('role');

        // Permission logic
        if ($currentUserRole === 'leader') {
            // Leader can remove anyone except themselves
            if ($currentUser->id == $userToRemove->id) {
                return ApiResponse::failed(__("teams.cannot_remove_self"), 403);
            }
        } elseif ($currentUserRole === 'subleader') {
            // Subleader can only remove members
            if ($userToRemoveRole !== 'member') {
                return ApiResponse::failed(__("teams.subleader_remove_member_only"), 403);
            }
        } else {
            // Members cannot remove anyone
            return ApiResponse::failed(__("teams.no_permission_remove"), 403);
        }

        $team->users()->detach($userToRemove->id);
        // Send notification to the removed user
        $userToRemove->notify(new UserRemovedFromTeamNotification($team, $currentUser));
        return ApiResponse::success(__("teams.user_removed_success"));
    }

    public function memberRole(ChangeRoleRequest $request)
    {
        $data = $request->validated();
        $team = Team::findOrFail($data['team_id']);
        $currentUser = auth('api')->user();

        // Only leader can change roles
        if (!$team->isLeader($currentUser)) {
            return ApiResponse::failed(__("teams.only_leader_can_change_role"), 403);
        }

        // Find user by phone number
        $userToChange = User::where('mobile', $data['phone_number'])->first();
        if (!$userToChange) {
            return ApiResponse::failed(__("teams.phone_not_associated"), 400);
        }

        // Prevent changing own role
        if ($currentUser->id == $userToChange->id) {
            return ApiResponse::failed(__("teams.cannot_change_own_role"), 403);
        }

        // Check if user is a member of the team
        $teamUser = TeamUser::where(['team_id' => $team->id, 'user_id' => $userToChange->id])->first();
        if (!$teamUser) {
            return ApiResponse::failed(__("teams.not_member"), 400);
        }

        if ($data['role'] === 'leader') {
            $leaderTeam = TeamUser::where(['team_id' => $team->id, 'user_id' => $currentUser->id])->first();
            $leaderTeam->role = 'member';
            $leaderTeam->save();
        }

        // Update the role
        $teamUser->role = $data['role'];
        $teamUser->save();
        // Send notification to the user whose role was changed
        $userToChange->notify(new UserRoleChangedNotification($team, $currentUser, $data['role']));
        return ApiResponse::success(__("teams.user_role_updated_success"));
    }

    public function leaveTeam($id)
    {
        $team = Team::findOrFail($id);
        $currentUser = auth('api')->user();

        // Check if user is a member of the team
        $isMember = $team->users()->where('user_id', $currentUser->id)->exists();
        if (!$isMember) {
            return ApiResponse::failed(__("teams.not_member"), 400);
        }

        // Check if user is a leader
        if ($team->isLeader($currentUser)) {
            return ApiResponse::failed(__("teams.cannot_remove_self"), 403);
        }

        // Remove user from team
        $team->users()->detach($currentUser->id);
        return ApiResponse::success(__("teams.user_left_success"));
    }
}
