<?php

namespace Modules\Teams\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Modules\Teams\Models\Team;
use Modules\User\Models\User;
use Modules\Common\Controllers\Admin\HelperController;
use Modules\Areas\Models\Area;
use App\Helpers\ApiResponse;

class AdminController extends HelperController
{
    protected $title = "Team";
    protected $switches = [];
    protected $links = [];

    public function __construct()
    {
        $this->model = new Team();
        $this->title = "Team";
        $this->name = 'teams';
        $this->list = [
            'name' => __('name'),
            'bio' => __('bio'),
        ];
        $this->switches['status'] = route('admin.teams.active_status');
    }

    public function listBuilder()
    {
        $this->list = [
            'name' => __('name'),
            'bio' => __('bio'),
        ];

        $this->links = [
            [
                'title' => __('members'),
                'icon' => 'fa fa-eye',
                'url' => route('admin.teams.members', "id"),
                'key' => 'id',
                'type' => 'secondary',
            ]
        ];
    }

    public function formBuilder()
    {
        $areas = Area::pluck('name', 'id')->toArray();
        $this->inputs = [
            'name' => ['title' => __('name')],
            'area_id' => ['title' => __('area'), 'type' => 'select', 'values' => $areas],
            'bio' => ['title' => __('bio')],
            'logo' => ['title' => __('logo'), 'type' => 'file', 'is_required' => ''],
        ];
    }

    public function store(Request $request)
    {
        $data = $request->all();
        if ($request->hasFile('logo')) {
            $uuid = (string) Str::uuid();
            $file = $request->file('logo');
            $ext = $file->getClientOriginalExtension();
            $filename = "{$uuid}.{$ext}";
            $path = $file->storeAs('teams', $filename, 'public');
            $data['logo'] = $path;
        }
        $team = Team::create($data);

        return redirect()->route('admin.teams.index');
    }

    public function update(Request $request, $id)
    {
        $team = Team::findOrFail($id);
        $data = $request->all();
        if ($request->hasFile('logo')) {
            if ($team->logo) {
                Storage::disk('public')->delete($team->logo);
            }
            $uuid = (string) Str::uuid();
            $file = $request->file('logo');
            $ext = $file->getClientOriginalExtension();
            $filename = "{$uuid}.{$ext}";
            $path = $file->storeAs('teams', $filename, 'public');
            $data['logo'] = $path;
        }
        $team->update($data);

        return $this->successfullResponse();
    }

    public function destroy($id)
    {
        $team = Team::findOrFail($id);
        if ($team->logo) {
            Storage::disk('public')->delete($team->logo);
        }
        $team->delete();

        return $this->successfullResponse();
    }

    public function active_status(Request $request)
    {
        $team = Team::findOrFail($request->id);
        $team->update(['status' => !$team->status]);
        return ApiResponse::get('', ['status' => 1]);
    }

    public function members(Request $request)
    {
        $teamId = $request->get('id') ?? $request->route('id');
        $team = Team::with(['users'])->findOrFail($teamId);
        $allUsers = User::get();
        $roles = [
            'leader' => __('Leader'),
            'subLeader' => __('Subleader'),
            'member' => __('Member'),
        ];
        return view('Teams::members', [
            'team' => $team,
            'allUsers' => $allUsers,
            'roles' => $roles,
            'title' => __('teams.Manage Members'),
        ]);
    }


    public function addMember(Request $request, $id)
    {
        $team = Team::findOrFail($id);
        $userName = $request->input('user_phone');
        $userId = User::where('mobile', $userName)->pluck('id')->first();
        if (!$userId) {
            return redirect()->back()->with('error', __('User with this phone number does not exist.'));
        }
        $role = $request->input('role');
        if ($role === 'leader') {
            $hasLeader = $team->users()->wherePivot('role', 'leader')->exists();
            if ($hasLeader) {
                return redirect()->back()->with('error', __('There is already a leader in this team.'));
            }
        }
        if (!$team->users->pluck('id')->contains($userId)) {
            $team->users()->attach($userId, ['role' => $role]);
            return redirect()->back();
        }
        return redirect()->back()->with('info', __('User is already a member of the team.'));
    }

    public function removeMembers(Request $request, $id)
    {
        $team = Team::findOrFail($id);
        $userIds = $request->input('user_ids', []);
        foreach ($userIds as $userId) {
            $team->users()->detach($userId);
        }
        return redirect()->back()->with('success', __('User has been removed from the team successfully'));
    }


    public function updateRoles(Request $request, $id)
    {
        $team = Team::findOrFail($id);
        $members = $request->input('members', []);

        // Check if there is already a leader in the team
        $hasLeader = $team->users()->wherePivot('role', 'leader')->exists();
        $assigningLeader = false;
        $assignLeaderUserId = null;
        foreach ($members as $userId => $role) {
            if ($role === 'leader') {
                $assigningLeader = true;
                $assignLeaderUserId = $userId;
                break;
            }
        }

        if ($assigningLeader && $hasLeader) {
            return back()->with('error', __('There is already a leader in this team.'));
        }

        foreach ($members as $userId => $role) {
            $team->users()->updateExistingPivot($userId, ['role' => $role]);
        }
        return back()->with('success', __('Roles updated successfully.'));
    }
}
