<?php

namespace Modules\Messages\Controllers;

use App\Helpers\ApiResponse;
use Illuminate\Http\Request;
use Modules\User\Models\User;
use Modules\Teams\Models\Team;
use App\Support\Actions\FCMAction;
use App\Http\Controllers\Controller;
use Modules\Messages\Models\Message;
use App\Services\GroupNotificationService;
use App\Notifications\ChatMessageNotification;
use Modules\Messages\Resources\MessageResource;
use App\Notifications\MessageUpdateNotification;

class ApiController extends Controller
{
    public function sendMessage(Request $request)
    {
        $request->validate([
            'team_id' => 'required|exists:teams,id',
            'content' => 'required|string|max:500',
        ]);

        $user = auth('api')->user();
        $team = Team::findOrFail($request->team_id);

        if (!$user->teams()->where('team_id', $team->id)->exists()) {
            return ApiResponse::failed(__('messages.You are not a member of this team'));
        }

        $message = Message::create([
            'user_id' => $user->id,
            'team_id' => $team->id,
            'content' => $request->content,
        ]);

        $teamMembers = $team->users()->where('team_user.user_id', '!=', $user->id)->get();

        foreach ($teamMembers as $member) {
            $member->notify(new ChatMessageNotification($team, $user, $message));
        }

        return ApiResponse::created([
            'message' => new MessageResource($message->load('user'))
        ]);
    }

    public function deleteMessage($messageId)
    {
        $user = auth('api')->user();
        $message = Message::findOrFail($messageId);

        if (!$user->teams()->where('team_id', $message->team_id)->exists()) {
            return ApiResponse::failed(__('You are not a member of this team'));
        }

        if (
            $message->user_id !== $user->id &&
            !$user->leaderTeams()->where('team_id', $message->team_id)->exists() &&
            !$user->subLeaderTeams()->where('team_id', $message->team_id)->exists()
        ) {
            return ApiResponse::failed(__('You can only delete your own messages or you must be a team leader/subleader'));
        }

        $message->delete();

        return ApiResponse::deleted(200, __('Message deleted successfully'));
    }

    public function getTeamMessages(Request $request, $teamId)
    {
        $user = auth('api')->user();
        $team = Team::findOrFail($teamId);

        if (!$user->teams()->where('team_id', $team->id)->exists()) {
            return ApiResponse::failed(__('You are not a member of this team'));
        }

        $messages = $team->messages()
            ->with(['user'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return ApiResponse::loaded([
            'messages' => MessageResource::collection($messages)
        ]);
    }
}
