<?php

namespace Modules\Match\Models;

use Carbon\Carbon;
use App\Traits\HasActive;
use Modules\User\Models\User;
use App\Enum\SubscriberStatus;
use Modules\Teams\Models\Team;
use App\Traits\HelperModelTrait;
use Illuminate\Database\Eloquent\Model;
use Modules\Playground\Models\Playground;
use Illuminate\Database\Eloquent\SoftDeletes;
use Modules\Challenges\Models\TeamMatchResult;

class Matche extends Model
{
    use HelperModelTrait, SoftDeletes, HasActive;
    protected $table = 'matches';
    protected $fillable = [
        'playground_id',
        'subscribers_qty',
        'date',
        'start_time',
        'end_time',
        'durations',
        'durations_text',
        'details',
        'amount',
        'type',
        'supervisor_id',
        'user_id',
        'status',
        'revision',
        'team1_id',
        'team2_id',
        'is_competitive'
    ];

    protected $casts = [
        'is_competitive' => 'boolean'
    ];

    public function playground()
    {
        return $this->belongsTo(Playground::class);
    }

    public function supervisor()
    {
        return $this->belongsTo(User::class, 'supervisor_id', 'id');
    }
    public function subscribers()
    {
        return $this->belongsToMany(User::class, 'subscribers', 'matche_id', 'user_id')->withPivot('created_at', 'id', 'status', 'user_id', 'payment_method', 'presence');
    }
    public function subscriber_lists()
    {
        return $this->hasMany(Subscriber::class, 'matche_id');
    }

    public function team1()
    {
        return $this->belongsTo(Team::class, 'team1_id');
    }

    public function team2()
    {
        return $this->belongsTo(Team::class, 'team2_id');
    }

    public function teamResults()
    {
        return $this->hasMany(TeamMatchResult::class, 'match_id');
    }

    public function scopeInTime($query)
    {
        return $query->where(function ($q) {
            $q->where(fn($qu) => $qu->whereDate('date', '>', Carbon::now()->format('Y-m-d')));
            $q->orWhere(fn($qu) => $qu->whereDate('date', '=', Carbon::now()->format('Y-m-d'))->whereTime('start_time', '>', Carbon::now()->format('H:i:s')));
        })->active();
    }
    public function scopeCurrentTime($query)
    {
        return $query->where(function ($q) {
            $q->where(fn($qu) => $qu->whereDate('date', '>', Carbon::now()->format('Y-m-d')));
            $q->orWhere(fn($qu) => $qu->whereDate('date', '=', Carbon::now()->format('Y-m-d'))->whereTime('start_time', '<', Carbon::now()->format('H:i:s'))->whereTime('end_time', '>', Carbon::now()->format('H:i:s')));
        })->active();
    }
    public function scopeOutTime($query)
    {
        return $query->where(function ($q) {
            $q->where(fn($qu) => $qu->whereDate('date', '<', Carbon::now()->format('Y-m-d')));
            $q->orWhere(fn($qu) => $qu->whereDate('date', '=', Carbon::now()->format('Y-m-d'))->whereTime('start_time', '<', Carbon::now()->format('H:i:s'))->whereTime('end_time', '<', Carbon::now()->format('H:i:s')));
        })->active();
    }

    public function scopeActive($query)
    {
        return $query->where('matches.status', true);
    }
    public function scopeActiveSubscribersCount($query)
    {
        return $query->withCount(['subscribers' => fn($q) => $q->where('subscribers.status', SubscriberStatus::IN_PROGRESS)]);
    }

    public function scopeFilter($query)
    {
        return $query->when(request('area_id'), fn($q) => $q->whereHas('playground', fn($q) => $q->where('playgrounds.area_id', request('area_id'))))->when(request('playgrounds'), fn($q) => $q->whereIn('playground_id', request('playgrounds')))->when(request('date'), fn($q) => $q->whereIn('date', request('date')));
    }


    public function scopeNearest($query, $lat, $lng)
    {
        if ($lat && $lng) {
            $distanceQuery = "( 6371 * acos( cos( radians(?) ) *
            cos( radians( JSON_UNQUOTE(JSON_EXTRACT(location, '$.lat')) ) )
            * cos( radians( JSON_UNQUOTE(JSON_EXTRACT(location, '$.lng')) ) - radians(?) )
            + sin( radians(?) ) *
            sin( radians( JSON_UNQUOTE(JSON_EXTRACT(location, '$.lat')) ) ) ) )
            AS distance";
            return $query->addSelect(['distance' => Playground::selectRaw($distanceQuery, [$lat, $lng, $lat])->whereColumn('matches.playground_id', 'playgrounds.id')->limit(1)])->orderBy('distance', 'asc');
        }
    }

    /**
     * Determine if the match is in the past (date and end_time before now)
     * @return bool
     */
    public function isPast()
    {
        if (!$this->date || !$this->end_time) {
            return false;
        }
        $endDateTime = \Carbon\Carbon::parse($this->date . ' ' . $this->end_time);
        return $endDateTime->lt(now());
    }
}
