<?php

namespace Modules\Match\Controllers;

use App\Enum\MatchType;
use App\Enum\SubscriberStatus;
use App\Enum\UserType;
use App\Helpers\ApiResponse;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\Matches\ApiStore;
use App\Notifications\MatchSubscribeNotifications;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;
use Illuminate\Validation\ValidationException;
use Modules\Match\Models\Matche;
use Modules\Match\Models\Subscriber;
use Modules\Match\Requests\AddSubscriber;
use Modules\Match\Requests\Subscribe;
use Modules\Match\Resources\MatcheResource;
use Modules\Match\Resources\SingleMatcheResource;
use Modules\Match\Resources\SingleSubscriberResource;
use Modules\Match\Resources\SubscriberResource;
use Modules\User\Models\User;
use MshMsh\Helpers\ApiResponder;

class ApiController extends Controller
{
    public function store(ApiStore $request)
    {
        Matche::create($request->all());
        return ApiResponse::created();
    }
    public function update($id, ApiStore $request)
    {
        $match = Matche::where('supervisor_id', auth()->id())->findOrFail($id)->loadCount('subscribers');
        throw_if(!is_null($match->user_id) || $match->subscribers_count > 0, ValidationException::withMessages(['msg' => __('this match is already reserved')]));
        $match->update($request->all());
        return ApiResponse::updated();
    }
    public function destroy($id)
    {
        $match = Matche::where('supervisor_id', auth()->id())->findOrFail($id)->loadCount('subscribers');
        throw_if((!is_null($match->user_id) || $match->subscribers_count > 0) && $match->end_time > Carbon::now()->format('Y-m-d'), ValidationException::withMessages(['msg' => __('this match is already reserved')]));
        $match->delete();
        return ApiResponse::deleted();
    }
    public function show($id)
    {
        $match = Matche::where('id', $id)->with(['subscribers', 'playground'])->withCount('subscriber_lists')->activeSubscribersCount()->firstOrFail();
        return ApiResponse::loaded([
            'match' => SingleMatcheResource::make($match)
        ]);
    }
    public function subscribersList($id)
    {
        $user = auth('api')->user();
        $match = Matche::where('id', $id)->when($user && $user->type == UserType::SUPERVISOR, fn($q) => $q->where('supervisor_id', $user->id))->with(['subscribers' => fn($q) => $q->where('subscribers.status', SubscriberStatus::IN_PROGRESS), 'subscribers.location'])->firstOrFail();
        if ($match->type == MatchType::SINGLE) {
            $subscribers = $match->subscribers;
        } else {
            $subscribers = Subscriber::where('matche_id', $match->id)->get();
        }
        return ApiResponse::loaded([
            'subscribers' => SubscriberResource::collection($subscribers)
        ]);
    }
    public function subscriber($id)
    {
        $subscriber = User::where('id', $id)->active()->with('area')->firstOrFail();
        return ApiResponse::loaded([
            'subscriber' => SingleSubscriberResource::make($subscriber)
        ]);
    }
    public function presence($id, Request $request)
    {
        $request->validate([
            'subscriber_id' => 'required',
            'payment_method' => 'required|in:cash,coupon,signal',
        ]);
        $user = auth('api')->user();
        $match = Matche::where('id', $id)->active()->when($user && $user->type == UserType::SUPERVISOR, fn($q) => $q->where('supervisor_id', $user->id))->firstOrFail();
        $subscriber = Subscriber::where('matche_id', $match->id)->where('user_id', $request->subscriber_id)->firstOrFail();
        $subscriber->update([
            'presence' => true,
            'payment_method' => $request['payment_method'],
        ]);
        return ApiResponse::loaded();
    }

    public function subscribe(Subscribe $request)
    {
        $match = Matche::where('id', $request['match_id'])->active()->inTime()->with(['subscribers'])->activeSubscribersCount()->first();
        throw_if(!$match, ValidationException::withMessages(['msg' => __('match is not exist or expired')]));
        $user = auth('api')->user();
        throw_if($match->type == MatchType::SINGLE && $match->subscribers()->where('subscribers.user_id', $user->id)->first(), ValidationException::withMessages(['msg' => __('you already subscribed to this match')]));
        throw_if($match->type == MatchType::GROUP && $match->user_id == $user->id, ValidationException::withMessages(['msg' => __('you already subscribed to this match')]));
        throw_if($match->type == MatchType::GROUP && !is_null($match->user_id), ValidationException::withMessages(['msg' => __('this match is already reserved')]));
        if ($match->type == MatchType::SINGLE) {
            if ($match->subscribers_count >= $match->subscribers_qty) {
                $match->subscribers()->syncwithoutdetaching([
                    $user->id => [
                        'created_at' => Carbon::now(),
                        'status' => SubscriberStatus::PENDING,
                    ]
                ]);
            } else {
                $match->subscribers()->syncwithoutdetaching([
                    $user->id => [
                        'created_at' => Carbon::now()
                    ]
                ]);
            }
        } else {
            $match->update(['user_id' => $user->id]);
        }
        return ApiResponse::created();
    }
    public function add_subscriber(AddSubscriber $request)
    {
        $match = Matche::where('id', $request['match_id'])->active()->inTime()->with(['subscribers'])->activeSubscribersCount()->first();
        throw_if(!$match, ValidationException::withMessages(['msg' => __('match is not exist or expired')]));
        throw_if($match->type == MatchType::SINGLE, ValidationException::withMessages(['msg' => __('this match is single you can not add subscriber')]));
        $exists = Subscriber::where('phone', $request->phone)->where('matche_id', $match->id)->exists();
        throw_if($exists, ValidationException::withMessages(['msg' => __('this subscriber is already subscribed to this match')]));
        if ($match->subscribers_count >= $match->subscribers_qty) {
            Subscriber::create([
                'name' => $request->name,
                'phone' => $request->phone,
                'matche_id' => $match->id,
                'created_at' => Carbon::now(),
                'status' => SubscriberStatus::PENDING,
            ]);
        } else {
            Subscriber::create([
                'name' => $request->name,
                'phone' => $request->phone,
                'matche_id' => $match->id,
                'created_at' => Carbon::now(),
            ]);
        }
        return ApiResponse::created();
    }
    public function cancel(Subscribe $request)
    {
        $match = Matche::where('id', $request['match_id'])->active()->with(['subscribers', 'playground'])
            ->withCount(['subscribers' => fn($q) => $q->where('subscribers.status', SubscriberStatus::PENDING)])->first();
        throw_if(!$match, ValidationException::withMessages(['msg' => __('match is not exist or expired')]));
        $user = auth('api')->user();
        if ($match->type == MatchType::SINGLE) {
            $subscriber = $match->subscribers->where('id', $user->id)->first();
            throw_if(!$subscriber, ValidationException::withMessages(['msg' => __('you already unsubscribe to this match')]));
            $subscriber_status = $subscriber->pivot?->status;
            $match->subscribers()->detach($user->id);
            if ($match->subscribers_count > 0 && $subscriber_status != SubscriberStatus::PENDING) {
                $pending_subscriber = $match->subscribers()->where('subscribers.status', SubscriberStatus::PENDING)->first();
                if ($pending_subscriber) {
                    $match->subscribers()->syncwithoutdetaching([
                        $pending_subscriber->id => [
                            'status' => SubscriberStatus::IN_PROGRESS,
                        ]
                    ]);
                    Notification::send($pending_subscriber, new MatchSubscribeNotifications($match));
                }
            }
        } else {
            throw_if(is_null($match->user_id), ValidationException::withMessages(['msg' => __('you already unsubscribe to this match')]));
            $match->update(['user_id' => null]);
        }

        return ApiResponse::loaded();
    }
    public function myMatches()
    {
        $user = auth('api')->user();
        if ($user->type == UserType::SUPERVISOR) {
            $user = $user->load(['supervisor_matches' => fn($q) => $q->with('playground')->activeSubscribersCount()
                ->filter()->when(request()->header('lat') && request()->header('lng'), fn($q) => $q->nearest(request()->header('lat'), request()->header('lng')))
                ->when(request('type'), fn($q) => $q->where('type', request('type')))->latest('date')]);
            $matches = $user->supervisor_matches;
        } else {
            $matches = Matche::where(function ($query) {
                $query->where('matches.user_id', auth('api')->id())->orWhereHas('subscribers', fn($q) => $q->where('subscribers.user_id', auth('api')->id()));
            })->with('playground')->activeSubscribersCount()->when(request('type') && request('type') == 'finished', fn($q) => $q->outTime(), fn($q) => $q->currentTime())->latest('date')->get();
        }
        return ApiResponse::loaded([
            'matches' => MatcheResource::collection($matches)
        ]);
    }

    public function allChallangeMatches()
    {
        $matches = Matche::where('type', MatchType::CHALLENGE)
            ->whereNull('team1_id')
            ->whereNull('team2_id')
            ->whereDate('date', '>=', now()->toDateString())
            ->orderBy('date', 'asc')
            ->orderBy('start_time', 'asc')
            ->get();

        return ApiResponse::loaded([
            'matches' => MatcheResource::collection($matches)
        ]);
    }

    public function challengesIndex()
    {
        $matches = Matche::with(['team1', 'team2'])->where('type', MatchType::CHALLENGE)
            ->whereDate('date', '>=', now()->toDateString())
            ->orderBy('date', 'asc')
            ->orderBy('start_time', 'asc')
            ->get();

        return ApiResponse::loaded([
            'matches' => MatcheResource::collection($matches)
        ]);
    }
}
