<?php

namespace Modules\Challenges\Requests;

use Modules\Match\Models\Matche;
use Modules\Teams\Models\TeamUser;
use Illuminate\Foundation\Http\FormRequest;

class StoreTeamMatchRequest extends FormRequest
{
    public function authorize()
    {
        return true;
    }

    public function rules()
    {
        // Players are required only when a team is requesting to join (not when team1 is inviting)
        $isTeam1Inviting = $this->isTeam1Inviting();

        $rules = [
            'match_id' => 'required|integer|exists:matches,id',
            'invited_team_id' => 'required|integer|exists:teams,id',
        ];

        // If team1 is inviting, players are not required (they'll send it when accepting)
        // If another team is requesting to join, players are required
        if (!$isTeam1Inviting) {
            $rules['players'] = 'required|array|min:10';
            $rules['players.*'] = [
                'required',
                'exists:users,id,deleted_at,NULL',
                function ($attribute, $value, $fail) {
                    $teamId = $this->input('invited_team_id');
                    $exists = TeamUser::query()
                        ->where('team_id', $teamId)
                        ->where('user_id', $value)
                        ->exists();

                    if (!$exists) {
                        $fail("يرجي التاكد من وجود الفرد في الفريق");
                    }
                },
            ];
        } else {
            // Optional for invitations, but if provided, validate it
            $rules['players'] = 'nullable|array';
            $rules['players.*'] = [
                'nullable',
                'exists:users,id,deleted_at,NULL',
            ];
        }

        return $rules;
    }

    /**
     * Check if team1 is inviting another team
     */
    private function isTeam1Inviting()
    {
        $matchId = $this->input('match_id');
        $invitedTeamId = $this->input('invited_team_id');

        if (!$matchId) {
            return false;
        }

        $match = Matche::find($matchId);
        if (!$match || !$match->team1_id) {
            return false;
        }

        $user = $this->user();
        if (!$user) {
            return false;
        }

        $requesterTeam = \Modules\Teams\Models\Team::find($match->team1_id);
        if (!$requesterTeam) {
            return false;
        }

        // Check if user is leader of team1 and inviting another team
        return $requesterTeam->isLeader($user) && $invitedTeamId != $match->team1_id;
    }

    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            $matchId = $this->input('match_id');
            $invitedTeamId = $this->input('invited_team_id');
            $requesterTeamId = null;
            if (method_exists($this, 'user') && $this->user() && property_exists($this->user(), 'team_id')) {
                $requesterTeamId = $this->user()->team_id;
            } elseif ($this->input('requester_team_id')) {
                $requesterTeamId = $this->input('requester_team_id');
            }

            if ($matchId && $requesterTeamId && $invitedTeamId) {
                $exists = \Modules\Challenges\Models\TeamMatchRequest::where('match_id', $matchId)
                    ->where(function ($q) use ($requesterTeamId, $invitedTeamId) {
                        $q->where(function ($q2) use ($requesterTeamId, $invitedTeamId) {
                            $q2->where('requester_team_id', $requesterTeamId)
                                ->where('invited_team_id', $invitedTeamId);
                        })->orWhere(function ($q2) use ($requesterTeamId, $invitedTeamId) {
                            $q2->where('requester_team_id', $invitedTeamId)
                                ->where('invited_team_id', $requesterTeamId);
                        });
                    })
                    ->whereIn('status', ['pending', 'accepted'])
                    ->exists();

                if ($exists) {
                    $validator->errors()->add('match_id', 'يوجد طلب معلق أو مقبول بالفعل بين هذين الفريقين لهذه المباراة.');
                }
            }
        });
    }
}
