<?php

namespace Modules\Challenges\Controllers;

use App\Helpers\ApiResponse;
use Illuminate\Http\Request;
use Modules\User\Models\User;
use App\Enum\SubscriberStatus;
use Modules\Teams\Models\Team;
use Modules\Match\Models\Matche;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Modules\Match\Models\Subscriber;
use Modules\Teams\Models\TeamRanking;
use Modules\Match\Resources\MatcheResource;
use Modules\Challenges\Models\TeamMatchResult;
use Modules\Challenges\Models\TeamMatchRequest;
use Modules\Challenges\Requests\BookMatchRequest;
use App\Notifications\TeamMatchRequestNotification;
use Modules\Challenges\Enum\TeamMatchRequestStatus;
use Modules\Challenges\Enum\TeamMatchRequestInitiator;
use Modules\Challenges\Requests\StoreTeamMatchRequest;
use Modules\Challenges\Requests\RespondToTeamMatchRequest;
use Modules\Challenges\Resources\TeamMatchRequestResource;
use Modules\Challenges\Requests\WithdrawTeamFromMatchRequest;

class ApiController extends Controller
{
    private function checkMatchNotPast($match)
    {
        if ($match->isPast()) {
            return ApiResponse::failed(__('messages.This match is in the past and cannot be modified'), 400);
        }
        return null;
    }

    public function getChallengeAvailableMatches($playgroundId)
    {
        $matches = Matche::where('type', 'challenge')
            ->where('playground_id', $playgroundId)
            ->whereNull('team1_id')
            ->active()
            ->inTime()
            ->orderBy('date', 'asc')
            ->orderBy('start_time', 'asc')
            ->get();
        return ApiResponse::loaded([
            'matches' => $matches,
        ]);
    }

    public function bookMatch(BookMatchRequest $request)
    {
        $user = auth()->user();
        $data = $request->validated();
        $team = Team::findOrFail($data['team_id']);
        $match = Matche::findOrFail($data['match_id']);

        if ($resp = $this->checkMatchNotPast($match)) return $resp;

        $activePlayersCount = $team->users()->where('users.status', true)->count();

        if ($activePlayersCount <= 10) {
            return ApiResponse::failed(__("teams.min_players"), 400);
        }

        if (!$team->isLeader($user)) {
            return ApiResponse::failed(__('teams.not_leader_of_team'), 403);
        }

        if (!is_null($match->team1_id)) {
            return ApiResponse::failed(__('teams.match_already_booked'), 400);
        }

        if ($match->type !== 'challenge') {
            return ApiResponse::failed(__('teams.not_challenge_type'), 400);
        }

        if ($match->status != 1) {
            return ApiResponse::failed(__('teams.match_not_active'), 400);
        }

        DB::transaction(function () use ($team, $match, $data) {
            // Update match with team1_id and is_competitive
            $match->team1_id = $team->id;
            $match->is_competitive = $data['is_competitive'];
            $match->save();

            // Get all unique active players of the team with their name and phone
            $activePlayers = $team->users()->whereIn('users.id', $data['players'])->where('users.status', true)->get(['users.id', 'users.name', 'users.mobile']);

            // Prepare bulk insert data for subscribers
            $now = now();
            $insertData = $activePlayers->unique('id')->map(function ($player) use ($match, $now) {
                return [
                    'user_id' => $player->id,
                    'matche_id' => $match->id,
                    'status' => SubscriberStatus::PENDING,
                    'name' => $player->name,
                    'phone' => $player->mobile,
                    'updated_at' => $now,
                    'created_at' => $now,
                ];
            })->toArray();

            // Insert all in one query, ignore duplicates
            Subscriber::insertOrIgnore($insertData);
        });

        return ApiResponse::success(__('teams.match_booked_success'));
    }

    public function withdrawTeamFromMatch(WithdrawTeamFromMatchRequest $request)
    {
        $user = auth()->user();
        $data = $request->validated();
        $teamId = $data['team_id'];
        $matchId = $data['match_id'];

        $team = Team::findOrFail($teamId);
        $match = Matche::findOrFail($matchId);

        if ($resp = $this->checkMatchNotPast($match)) return $resp;

        // Only leader can withdraw
        if (!$team->isLeader($user)) {
            return ApiResponse::failed(__('teams.not_leader_of_team'), 403);
        }

        $isTeam1 = $match->team1_id == $team->id;
        $isTeam2 = $match->team2_id == $team->id;
        if (!$isTeam1 && !$isTeam2) {
            return ApiResponse::failed(__('teams.team_not_assigned'), 400);
        }

        DB::transaction(function () use ($match, $team, $isTeam1, $isTeam2) {
            if ($isTeam1) {
                // Remove both teams and all subscribers
                $match->team1_id = null;
                $match->team2_id = null;
                $match->is_competitive = false;
                $match->save();
                Subscriber::where('matche_id', $match->id)->delete();
            } elseif ($isTeam2) {
                // Remove only team2 and its subscribers
                $match->team2_id = null;
                $match->save();
                Subscriber::where('matche_id', $match->id)
                    ->whereIn('user_id', $team->users()->pluck('users.id'))
                    ->delete();
            }
        });

        return ApiResponse::success(__('teams.team_withdrawn_success'));
    }

    public function sendTeamMatchRequest(StoreTeamMatchRequest $request)
    {
        $user = auth()->user();
        $data = $request->validated();
        $match = Matche::findOrFail($data['match_id']);
        $invitedTeam = Team::findOrFail($data['invited_team_id']);
        $requesterTeam = Team::findOrFail($match->team1_id);
        $userTeams = $user->teams()->pluck('teams.id')->toArray();

        if ($resp = $this->checkMatchNotPast($match)) return ApiResponse::failed(__('teams.match_in_past'), 400);

        // If user is leader of team1 (invite another team)
        if ($requesterTeam->isLeader($user)) {
            // Prevent inviting self
            if ($invitedTeam->id == $requesterTeam->id) {
                return ApiResponse::failed(__('teams.team_cannot_invite_self'), 400);
            }
            // Check if invited team has at least 10 active members
            $activePlayersCount = $invitedTeam->users()->where('users.status', true)->count();
            if ($activePlayersCount < 10) {
                return ApiResponse::failed(__("teams.min_players"), 400);
            }
            // Prevent duplicate requests
            if (TeamMatchRequest::where('match_id', $match->id)->where('invited_team_id', $invitedTeam->id)->where('status', TeamMatchRequestStatus::PENDING)->exists()) {
                return ApiResponse::failed(__('teams.pending_request_exists'), 400);
            }
            // When team1 invites, we don't store players array - invited team will send it when accepting
            $teamMatchRequest = DB::transaction(function () use ($match, $requesterTeam, $invitedTeam) {
                $req = TeamMatchRequest::create([
                    'match_id' => $match->id,
                    'requester_team_id' => $requesterTeam->id,
                    'invited_team_id' => $invitedTeam->id,
                    'status' => TeamMatchRequestStatus::PENDING,
                    'initiated_by' => TeamMatchRequestInitiator::REQUESTER,
                    'invited_team_players' => null, // Not stored when invitation is sent
                ]);
                return $req;
            });
            // Notify the leader of the invited team
            $leader = $invitedTeam->users()->wherePivot('role', 'leader')->first();
            if ($leader) {
                $leader->notify(new TeamMatchRequestNotification($teamMatchRequest, 'created', $user));
            }
            return ApiResponse::success(__('teams.team_match_request_sent'));
        }

        // If user is leader of another team (not team1), and team2 is null, allow join request
        $userTeam = null;
        foreach ($userTeams as $tid) {
            $team = Team::find($tid);
            if ($team && $team->isLeader($user) && $team->id != $match->team1_id) {
                $userTeam = $team;
                break;
            }
        }

        if ($userTeam && is_null($match->team2_id)) {
            // Check if user team has at least 10 active members
            $activePlayersCount = $userTeam->users()->where('users.status', true)->count();
            if ($activePlayersCount < 10) {
                return ApiResponse::failed(__("teams.min_players"), 400);
            }
            // Prevent duplicate requests
            if (TeamMatchRequest::where('match_id', $match->id)->where('invited_team_id', $userTeam->id)->where('status', TeamMatchRequestStatus::PENDING)->exists()) {
                return ApiResponse::failed(__('teams.pending_request_exists'), 400);
            }
            $teamMatchRequest = DB::transaction(function () use ($match, $requesterTeam, $userTeam, $data) {
                $req = TeamMatchRequest::create([
                    'match_id' => $match->id,
                    'requester_team_id' => $requesterTeam->id, // team1 is always the requester
                    'invited_team_id' => $userTeam->id,
                    'status' => TeamMatchRequestStatus::PENDING,
                    'initiated_by' => TeamMatchRequestInitiator::INVITED,
                    'invited_team_players' => $data['players'] ?? null,
                ]);
                return $req;
            });
            // Notify the leader of team1
            $leader = $requesterTeam->users()->wherePivot('role', 'leader')->first();
            if ($leader) {
                $leader->notify(new TeamMatchRequestNotification($teamMatchRequest, 'created', $user));
            }
            return ApiResponse::success(__('teams.join_request_sent'));
        }

        return ApiResponse::failed(__('teams.not_authorized_send_request'), 403);
    }

    /**
     * Accept or reject a team match request
     */
    public function respondToTeamMatchRequest(RespondToTeamMatchRequest $request)
    {
        $user = auth()->user();
        $data = $request->validated();
        $teamMatchRequest = TeamMatchRequest::findOrFail($data['request_id']);
        $match = Matche::findOrFail($teamMatchRequest->match_id);
        $invitedTeam = Team::findOrFail($teamMatchRequest->invited_team_id);
        $requesterTeam = Team::findOrFail($teamMatchRequest->requester_team_id);

        // Only leader of the correct team can respond
        if ((string)$teamMatchRequest->initiated_by === TeamMatchRequestInitiator::REQUESTER->value) {
            // If team1 invited, only invited team leader can respond
            if (!$invitedTeam->isLeader($user)) {
                return ApiResponse::failed(__('teams.only_leader_invited_team'), 403);
            }
        } else {
            // If team requested to join, only team1 leader can respond
            if (!$requesterTeam->isLeader($user)) {
                return ApiResponse::failed(__('teams.only_leader_team1'), 403);
            }
        }

        // Check if match is in the past
        if ($resp = $this->checkMatchNotPast($match)) return $resp;

        if ($teamMatchRequest->status !== TeamMatchRequestStatus::PENDING->value) {
            return ApiResponse::failed(__('teams.request_already_responded'), 400);
        }

        if ($data['action'] === 'accept') {
            // Accept: set team2_id and add selected players as subscribers
            $selectedPlayerIds = null;

            // Determine where to get players from based on who initiated the request
            if ((string)$teamMatchRequest->initiated_by === TeamMatchRequestInitiator::REQUESTER->value) {
                // If team1 invited (REQUESTER initiated), invited team sends players when accepting
                if (empty($data['players'])) {
                    return ApiResponse::failed(__("teams.no_players_selected"), 400);
                }
                $selectedPlayerIds = $data['players'];
            } else {
                // If team requested to join (INVITED initiated), players are already stored
                if (empty($teamMatchRequest->invited_team_players)) {
                    return ApiResponse::failed(__("teams.no_players_selected"), 400);
                }
                // Decode the JSON if it's a string, otherwise use as is
                $selectedPlayerIds = is_string($teamMatchRequest->invited_team_players)
                    ? json_decode($teamMatchRequest->invited_team_players, true)
                    : $teamMatchRequest->invited_team_players;
            }

            // Validate that we have at least 10 players
            if (count($selectedPlayerIds) < 10) {
                return ApiResponse::failed(__("teams.min_players"), 400);
            }

            DB::transaction(function () use ($match, $invitedTeam, $teamMatchRequest, $user, $selectedPlayerIds, $data) {
                $match->team2_id = $invitedTeam->id;
                $match->save();
                $teamMatchRequest->status = TeamMatchRequestStatus::ACCEPTED;

                // If this was an invitation (REQUESTER initiated), store the players array now
                if ((string)$teamMatchRequest->initiated_by === TeamMatchRequestInitiator::REQUESTER->value && !empty($data['players'])) {
                    $teamMatchRequest->invited_team_players = $data['players'];
                }

                $teamMatchRequest->save();

                // Add only the selected players as subscribers
                $selectedPlayers = $invitedTeam->users()
                    ->whereIn('users.id', $selectedPlayerIds)
                    ->where('users.status', true)
                    ->get(['users.id', 'users.name', 'users.mobile']);

                $now = now();
                $insertData = $selectedPlayers->unique('id')->map(function ($player) use ($match, $now) {
                    return [
                        'user_id' => $player->id,
                        'matche_id' => $match->id,
                        'status' => SubscriberStatus::PENDING,
                        'name' => $player->name,
                        'phone' => $player->mobile,
                        'updated_at' => $now,
                        'created_at' => $now,
                    ];
                })->toArray();

                Subscriber::insertOrIgnore($insertData);
            });

            // Notify the leader of the other team
            if ((string)$teamMatchRequest->initiated_by === TeamMatchRequestInitiator::REQUESTER->value) {
                // If team1 invited, notify team1 leader
                $leader = $requesterTeam->users()->wherePivot('role', 'leader')->first();
            } else {
                // If team requested to join, notify invited team leader
                $leader = $invitedTeam->users()->wherePivot('role', 'leader')->first();
            }

            if ($leader) {
                $leader->notify(new TeamMatchRequestNotification($teamMatchRequest, 'accepted', $user));
            }
            return ApiResponse::success(__('teams.team_match_request_accepted'));
        } else {
            // Reject
            $teamMatchRequest->status = TeamMatchRequestStatus::REJECTED;
            $teamMatchRequest->save();

            // Notify the leader of the other team
            if ((string)$teamMatchRequest->initiated_by === TeamMatchRequestInitiator::REQUESTER->value) {
                // If team1 invited, notify team1 leader
                $leader = $requesterTeam->users()->wherePivot('role', 'leader')->first();
            } else {
                // If team requested to join, notify invited team leader
                $leader = $invitedTeam->users()->wherePivot('role', 'leader')->first();
            }

            if ($leader) {
                $leader->notify(new TeamMatchRequestNotification($teamMatchRequest, 'rejected', $user));
            }
            return ApiResponse::success(__('teams.team_match_request_rejected'));
        }
    }

    public function getTeamMatchInvites(Request $request)
    {
        $user = auth()->user();
        $teamIds = $user->teams()->pluck('teams.id')->toArray();
        $leaderTeamIds = [];
        foreach ($teamIds as $tid) {
            $team = Team::find($tid);
            if ($team && $team->isLeader($user)) {
                $leaderTeamIds[] = $tid;
            }
        }
        if (empty($leaderTeamIds)) {
            return ApiResponse::failed(__('teams.not_a_leader_of_any_team'), 403);
        }
        // Get all requests where the user is leader of requester or invited team
        $requests = TeamMatchRequest::with(['match', 'requesterTeam', 'invitedTeam'])
            ->where(function ($q) use ($leaderTeamIds) {
                $q->whereIn('requester_team_id', $leaderTeamIds)
                    ->orWhereIn('invited_team_id', $leaderTeamIds);
            })
            ->orderByDesc('created_at')
            ->paginate(10);
        return ApiResponse::success(
            __('teams.invites_requests_fetched_successfully'),
            TeamMatchRequestResource::collection($requests)
        );
    }

    public function setTeamMatchResult(Request $request)
    {
        $request->validate([
            'match_id' => 'required|exists:matches,id',
            'results' => 'required|array|size:2',
            'results.*.team_id' => 'required|exists:teams,id',
            'results.*.goals' => 'required|integer|min:0',
        ]);

        $match = Matche::findOrFail($request->match_id);
        $is_competitive = $match->is_competitive;
        $teamIds = [$match->team1_id, $match->team2_id];
        $submittedTeamIds = [
            $request->results[0]['team_id'],
            $request->results[1]['team_id'],
        ];
        sort($teamIds);
        sort($submittedTeamIds);
        if ($teamIds !== $submittedTeamIds) {
            return ApiResponse::failed(__('messages.Submitted teams do not match the teams in this match.'), 422);
        }

        DB::transaction(function () use ($request, $is_competitive) {
            foreach ($request->results as $result) {
                TeamMatchResult::updateOrCreate(
                    [
                        'match_id' => $request->match_id,
                        'team_id' => $result['team_id'],
                        'is_competitive' => $is_competitive
                    ],
                    [
                        'goals' => $result['goals'],
                    ]
                );
            }
        });

        // Recalculate rankings for both teams
        $this->calculateTeamRankings($teamIds);

        return ApiResponse::success(__('messages.Results set successfully'));
    }

    public function getTeamMatchResults($id)
    {
        $results = TeamMatchResult::where('match_id', $id)->with('team')->get();
        return ApiResponse::success(__('messages.Results fetched successfully'), $results);
    }

    public function challengesOverview()
    {
        $teams = Team::where('status', true)
            ->with(['ranking' => function ($q) {
                $q->where('is_competitive', true);
            }])
            ->get();
        return ApiResponse::success(__('messages.Challenges overview fetched successfully'), $teams);
    }

    /**
     * Recalculate team rankings for the given team IDs
     */
    private function calculateTeamRankings(array $teamIds)
    {
        // Level order as per migration
        $levels = [
            'Bronze1',
            'Bronze2',
            'Bronze3',
            'Silver1',
            'Silver2',
            'Silver3',
            'Gold1',
            'Gold2',
            'Gold3',
            'Platinum1',
            'Platinum2',
            'Platinum3',
            'Legendary1',
            'Legendary2',
            'Legendary3',
            'Elite1',
            'Elite2',
            'Elite3',
            'Master'
        ];
        foreach ($teamIds as $teamId) {
            foreach ([true, false] as $is_competitive) {
                // Only count matches for this is_competitive value
                $results = TeamMatchResult::where('team_id', $teamId)
                    ->where('is_competitive', $is_competitive)
                    ->get();
                $matchCount = $results->count();
                $wins = $draws = $losses = $owns = $againsts = $points = 0;
                foreach ($results as $result) {
                    // Find the opponent's result for this match
                    $opponentResult = TeamMatchResult::where('match_id', $result->match_id)
                        ->where('team_id', '!=', $teamId)
                        ->where('is_competitive', $is_competitive)
                        ->first();
                    if (!$opponentResult) continue;
                    $owns += $result->goals;
                    $againsts += $opponentResult->goals;
                    if ($result->goals > $opponentResult->goals) {
                        $wins++;
                        $points += 3;
                    } elseif ($result->goals == $opponentResult->goals) {
                        $draws++;
                        $points += 1;
                    } else {
                        $losses++;
                    }
                }
                // Level up every 5 matches
                $levelIndex = min(intval($matchCount / 5), count($levels) - 1);
                $level = $levels[$levelIndex];
                // Update or create ranking for this team and is_competitive value
                TeamRanking::updateOrCreate(
                    ['team_id' => $teamId, 'is_competitive' => $is_competitive],
                    [
                        'wins' => $wins,
                        'draws' => $draws,
                        'losses' => $losses,
                        'owns' => $owns,
                        'againsts' => $againsts,
                        'level' => $level,
                        'points' => $points
                    ]
                );
            }
        }
    }
}
