<?php

namespace Modules\Challenges\Controllers;

use App\Enum\UserType;
use App\Enum\MatchType;
use Illuminate\Support\Str;
use App\Helpers\ApiResponse;
use Illuminate\Http\Request;
use Modules\User\Models\User;
use Modules\Teams\Models\Team;
use Modules\Match\Models\Matche;
use Modules\Match\Models\Subscriber;
use Modules\Teams\Models\TeamRanking;
use Modules\Playground\Models\Playground;
use Modules\Challenges\Models\TeamMatchResult;
use Modules\Challenges\Models\TeamMatchRequest;
use Modules\Common\Controllers\Admin\HelperController;

class AdminController extends HelperController
{
   protected $title = "Challenge Matches";
   protected $switches = [];
   protected $links = [];
   protected $rows;
   protected $formRequest;

   public function __construct()
   {
      $this->model = new Matche();
      $this->name = 'challenges';
      $this->myname = Str::singular('challenges');
      $this->title = "Challenge Matches";
      $this->canShow = true;

      $this->rows = $this->model->newQuery()
         ->where('type', MatchType::CHALLENGE)
         ->with(['playground', 'team1', 'team2']);
      // Add status switch for the list
      $this->switches['status'] = route('admin.challenges.active_status', [':id']);
   }

   public function listBuilder()
   {
      $this->list = [
         'playground_name' => __('playground'),
         'date' => __('date'),
         'start_time' => __('start time'),
         'end_time' => __('end time'),
         'amount' => __('amount'),
         'team1_name' => __('challenges.Team 1'),
         'team2_name' => __('challenges.Team 2'),
      ];
   }

   public function getTeams()
   {
      $name = request('term.term');

      $teams = Team::where('name', 'like', "%{$name}%")->get();
      $options = [];
      foreach ($teams as $team) {
         $options[] = [
            'id' => $team->id,
            'name' => $team->name,
         ];
      }
      return response()->json($options);
   }

   public function formBuilder()
   {
      $playgrounds = Playground::active()->pluck('name', 'id')->toArray();
      $supervisors = User::where('type', UserType::SUPERVISOR)->pluck('name', 'id')->toArray();

      // Get the model if available (for edit mode)
      $model = isset($this->model) ? $this->model : null;
      $team1 = null;
      $team2 = null;
      if ($model && $model->team1_id) {
         $team1 = Team::find($model->team1_id);
      }
      if ($model && $model->team2_id) {
         $team2 = Team::find($model->team2_id);
      }

      $this->inputs = [
         'playground_id' => ['title' => __('playground'), 'type' => 'select', 'values' => $playgrounds],
         'supervisor_id' => ['title' => __('supervisor'), 'type' => 'select', 'values' => $supervisors],
         'date' => ['title' => __('date'), 'type' => 'date'],
         'start_time' => ['title' => __('time'), 'type' => 'time'],
         'end_time' => ['title' => __('end time'), 'type' => 'time'],
         'subscribers_qty' => ['title' => __('subscribers count'), 'type' => 'number'],
         'durations' => ['title' => __('durations'), 'type' => 'number'],
         'durations_text' => ['title' => __('duration text'), 'type' => 'text'],
         'details' => ['title' => __('details'), 'type' => 'textarea'],
         'amount' => ['title' => __('amount'), 'type' => 'number'],
         'team1_id' => [
            'title' => __('challenges.Team 1'),
            'type' => 'ajax_select',
            'action' => route('admin.challenges.get_teams'),
            'values' => $team1 ? [$team1->id => $team1->name] : [],
            'empty' => 1,
            'default' => function ($model) {
               return $model ? $model->team1_id : null;
            }
         ],
         'team2_id' => [
            'title' => __('challenges.Team 2'),
            'type' => 'ajax_select',
            'action' => route('admin.challenges.get_teams'),
            'values' => $team2 ? [$team2->id => $team2->name] : [],
            'empty' => 1,
            'default' => function ($model) {
               return $model ? $model->team2_id : null;
            }
         ],
         'is_competitive' => [
            'title' => __('challenges.Is Competitive'),
            'type' => 'select',
            'values' => ['0' => 'No', '1' => 'Yes'],
            'default' => function ($model) {
               return $model ? (int)$model->is_competitive : 0;
            }
         ],
         'type' => ['title' => '', 'type' => 'hidden', 'value' => MatchType::CHALLENGE],
      ];
   }

   public function show()
   {
      $id = request('id');
      $match = Matche::with(['team1', 'team2', 'playground', 'teamResults'])->findOrFail($id);

      // Get team members with subscription data
      $team1Members = [];
      $team2Members = [];

      if ($match->team1) {
         $team1Members = $this->getTeamMembersWithSubscriptions($match->team1, $match->id);
      }

      if ($match->team2) {
         $team2Members = $this->getTeamMembersWithSubscriptions($match->team2, $match->id);
      }

      return view('Challenges::show', [
         'match' => $match,
         'team1Members' => $team1Members,
         'team2Members' => $team2Members,
         'title' => __('challenges.Challenge Match Details')
      ]);
   }

   /**
    * Get team members with their subscription status for a match
    */
   protected function getTeamMembersWithSubscriptions($team, $matchId)
   {
      $members = $team->users()->get();
      $data = [];

      foreach ($members as $user) {
         $subscriber = Subscriber::where('matche_id', $matchId)
            ->where('user_id', $user->id)
            ->first();

         $data[] = (object) [
            'user' => $user,
            'subscriber' => $subscriber
         ];
      }

      return $data;
   }

   public function store(Request $request)
   {
      $data = $request->all();
      if (!empty($data['team1_id']) && !empty($data['team2_id']) && $data['team1_id'] == $data['team2_id']) {
         return $this->failedfullResponse(__('teams.teams should be diffrent'));
      }
      $match = $this->model->create($data);
      $this->syncTeamSubscribers($match, $data);
      return $this->successfullResponse();
   }

   public function update(Request $request, $id)
   {
      $data = $request->all();
      if (!empty($data['team1_id']) && !empty($data['team2_id']) && $data['team1_id'] == $data['team2_id']) {
         return $this->failedfullResponse(__('teams.teams should be diffrent'));
      }

      $resultsExist = TeamMatchResult::where('match_id', $id)->exists();
      if ($resultsExist) {
         return $this->failedfullResponse(__('challenges.There are results for this match, you can\'t update the teams'));
      }

      $match = $this->model->findOrFail($id);
      $match->update($data);
      $this->syncTeamSubscribers($match, $data, true);
      return $this->successfullResponse();
   }

   protected function syncTeamSubscribers($match, $data, $isUpdate = false)
   {
      $newTeamIds = [];
      if (!empty($data['team1_id'])) $newTeamIds[] = $data['team1_id'];
      if (!empty($data['team2_id'])) $newTeamIds[] = $data['team2_id'];

      $newUsers = collect();
      foreach ($newTeamIds as $teamId) {
         $team = Team::find($teamId);
         if ($team) {
            $newUsers = $newUsers->merge($team->users()->get());
         }
      }
      $newUsers = $newUsers->unique('id');

      if ($isUpdate) {
         $match->subscribers()->detach();
      }

      if ($newUsers->isNotEmpty()) {
         $attachData = [];
         foreach ($newUsers as $user) {
            $attachData[$user->id] = [
               'created_at' => now(),
               'phone' => $user->mobile,
               'name' => $user->name,
               'status' => 'pending',
            ];
         }
         $match->subscribers()->syncWithoutDetaching($attachData);
      }
   }

   /**
    * Toggle subscription status for a user in a match
    */
   public function toggleSubscription(Request $request)
   {
      $request->validate([
         'match_id' => 'required|exists:matches,id',
         'user_id' => 'required|exists:users,id',
         'action' => 'required|in:subscribe,unsubscribe'
      ]);

      $matchId = $request->match_id;
      $userId = $request->user_id;
      $action = $request->action;

      if ($action === 'subscribe') {
         $user = User::findOrFail($userId);
         Subscriber::updateOrCreate(
            ['matche_id' => $matchId, 'user_id' => $userId],
            [
               'phone' => $user->mobile,
               'name' => $user->name,
               'status' => 'pending',
            ]
         );
      } elseif ($action === 'unsubscribe') {
         Subscriber::where('matche_id', $matchId)
            ->where('user_id', $userId)
            ->delete();
      }

      return redirect()->route('admin.challenges.show', $matchId)
         ->with('success', __('challenges.Subscription updated successfully'));
   }

   protected function calculateTeamRankings(array $teamIds)
   {
      $levels = [
         'Bronze1',
         'Bronze2',
         'Bronze3',
         'Silver1',
         'Silver2',
         'Silver3',
         'Gold1',
         'Gold2',
         'Gold3',
         'Platinum1',
         'Platinum2',
         'Platinum3',
         'Legendary1',
         'Legendary2',
         'Legendary3',
         'Elite1',
         'Elite2',
         'Elite3',
         'Master'
      ];
      foreach ($teamIds as $teamId) {
         foreach ([true, false] as $is_competitive) {
            $results = TeamMatchResult::where('team_id', $teamId)
               ->where('is_competitive', $is_competitive)
               ->get();
            $matchCount = $results->count();
            $wins = $draws = $losses = $owns = $againsts = $points = 0;
            foreach ($results as $result) {
               $opponentResult = TeamMatchResult::where('match_id', $result->match_id)
                  ->where('team_id', '!=', $teamId)
                  ->where('is_competitive', $is_competitive)
                  ->first();
               if (!$opponentResult) continue;
               $owns += $result->goals;
               $againsts += $opponentResult->goals;
               if ($result->goals > $opponentResult->goals) {
                  $wins++;
                  $points += 3;
               } elseif ($result->goals == $opponentResult->goals) {
                  $draws++;
                  $points += 1;
               } else {
                  $losses++;
               }
            }
            $levelIndex = min(intval($matchCount / 5), count($levels) - 1);
            $level = $levels[$levelIndex];
            TeamRanking::updateOrCreate(
               ['team_id' => $teamId, 'is_competitive' => $is_competitive],
               [
                  'wins' => $wins,
                  'draws' => $draws,
                  'losses' => $losses,
                  'owns' => $owns,
                  'againsts' => $againsts,
                  'level' => $level,
                  'points' => $points
               ]
            );
         }
      }
   }

   public function updateResults(Request $request, $id)
   {
      $request->validate([
         'results' => 'required|array|size:2',
         'results.*.team_id' => 'required|exists:teams,id',
         'results.*.goals' => 'required|integer|min:0',
      ]);

      $match = Matche::findOrFail($id);
      $is_competitive = $match->is_competitive;
      $teamIds = [$match->team1_id, $match->team2_id];
      $submittedTeamIds = [
         $request->results[0]['team_id'],
         $request->results[1]['team_id'],
      ];
      sort($teamIds);
      sort($submittedTeamIds);

      foreach ($request->results as $result) {
         TeamMatchResult::updateOrCreate(
            [
               'match_id' => $id,
               'team_id' => $result['team_id'],
               'is_competitive' => $is_competitive
            ],
            [
               'goals' => $result['goals'],
            ]
         );
      }

      $this->calculateTeamRankings($teamIds);

      return redirect()->route('admin.challenges.show', $id)->with('success', __('challenges.Results updated successfully'));
   }

   public function results($id)
   {
      $match = Matche::with(['team1', 'team2'])->findOrFail($id);
      $results = TeamMatchResult::where('match_id', $id)->get();
      return view('Challenges::results', [
         'match' => $match,
         'results' => $results,
         'title' => __('challenges.Match Results'),
      ]);
   }

   public function active_status(Request $request)
   {
      $match = Matche::findOrFail($request->id);
      $match->update(['status' => !$match->status]);
      return ApiResponse::get('', ['status' => 1]);
   }
}
