<?php

namespace Modules\User\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Mail;
use Modules\Orders\Models\Cart;
use Modules\Pages\Models\Page;
use Modules\User\Models\Token;
use Modules\User\Models\User;
use Modules\User\Models\UserLike;
use Modules\User\Services\Confirmation;

class WebController extends Controller
{
    public function register(Request $request)
    {
        if ($request->isMethod('GET')) {
            $terms = Page::where('type', 'terms')->first() ?? new Page();
            return view('User::auth.register', get_defined_vars());
        }
        $request->validate([
            'email' => 'required|unique:users,email',
            'mobile' => ['required', 'regex:/^05[0-9]{8}$/', 'unique:users,mobile'],
            'name' => 'required',
            'password' => 'required|confirmed',
            'accept_terms' => 'required'
        ]);
        $user = User::create($request->all());
        Confirmation::sendMail($user);
        if (request('fromCart')) session()->put('fromCart', 1);
        // auth()->login($user);
        return redirect()->route('user.active', Crypt::encrypt($request->email))->with('success', __('Verification code sent to your email'));
    }

    public function login(Request $request)
    {
        if (request()->isMethod('get')) {
            return abort('404');
        }
        if (
            Auth::attempt(['email' => $request->email, 'password' => $request->password], true) ||
            Auth::attempt(['mobile' => $request->email, 'password' => $request->password], true)
        ) {
            if (!auth()->user()->status) {
                return back()->with('error', __('Your account need to be activated'));
            }
            return redirect()->to('/')->with('success', __('Logined successfully'));
        }
        // dd('dd');
        return back()->with('error', __('Not correct email or password'));
    }

    public function active(Request $request, $token)
    {
        if ($request->isMethod('GET')) {
            return view('User::auth.activate', [
                'email' => Crypt::decrypt($token)
            ]);
        }
        $token = implode('', $request->code);
        $token = Token::where('token', $token)->latest()->first();
        if (!$token) {
            return redirect()->back()->with('error', __('Not valid active code'));
        }
        $token->user()->update(['status' => 1]);
        auth()->login($token->user);
        $token->delete();
        if (session('fromCart')) {
            Cart::where('ip', $_SERVER['REMOTE_ADDR'])->update(['user_id' => auth()->id()]);
            session()->forget('fromCart');
            return redirect()->route('cart.index');
        }
        return redirect()->to('/')->with('success', __('Your account activated successfully , you loginned successfully'));
    }

    public function resend_code(Request $request)
    {
        $user = session('user');
        $last_token_time = $user->token()->first()->created_at;
        if ($user && time() > strtotime($last_token_time . ' +2 minutes')) {
            $this->send_verify_code($user);
            return back()->with('success', __('verify'));
        }
        return back();
    }

    public function send_verify_code($user)
    {
        $code = rand(1000, 9999);
        $code = 1234;
        $user->token()->delete();
        $user->token()->firstOrCreate(['token' => $code]);
        $message = __("Your code is : ") . $code;
        if (session('to') == 'mobile' || request('mobile') == $user->mobile) {
            send_sms($user->mobile, $message);
            $to = "mobile";
        } else {
            $to = "email";
        }
        session()->put('user', $user);
        session()->put('to', $to);
    }


    public function logout()
    {
        auth()->logout();
        return redirect()->to('login');
    }
    public function verify(Request $request)
    {
        $user = session('user');
        $code = $request->code;
        if (is_array($code)) {
            $code = implode('', $code);
        }
        $token = $user->token()->where('token', $code)->first();
        if (!$token) {
            return back()->with('error', 'Verification code is not correct');
        }
        auth()->login($user);

        $cart = Cart::where('ip', request()->ip())->whereNull('user_id');
        if ($cart->count()) {
            $cart->update(['user_id' => $user->id]);
        }
        $likes = UserLike::where('ip', request()->ip())->whereNull('user_id');
        if ($likes->count()) {
            $likes->update(['user_id' => $user->id]);
        }
        SearchResult::where('ip', request()->ip())->whereNull('user_id')->update(['user_id' => $user->id]);
        return back();
    }
}
