<?php

namespace Modules\Payment\Services;

use App\Services\Services;
use Modules\User\Models\User;
use Modules\Payment\Models\Wallet;
use Illuminate\Support\Facades\DB;
use Exception;
use Illuminate\Support\Facades\Log;
use Stripe\Refund;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use Modules\Payment\Models\WalletDeposit;

class WalletService extends Services
{

    //For Full Payment For The Match
    public function pay(User $user, float $amount, $referenceModel = null, string $description = '')
    {

        return DB::transaction(function () use ($user, $amount, $referenceModel, $description) {
            $wallet = $user->wallet()->lockForUpdate()->first();


            if ($wallet->balance < $amount) {
                return $this->setError(' not Enough Balance Please Charge Your Wallet');
            }

            // 1. Log Transaction
            $transaction = $wallet->transactions()->create([
                'type' => 'payment',
                'amount' => -$amount,
                'balance_after' => $wallet->balance - $amount,
                'reference_id' => $referenceModel ? $referenceModel->id : null,
                'reference_type' => $referenceModel ? get_class($referenceModel) : null,
                'description' => $description,
            ]);

            // 2. Decrement Wallet Balance
            $wallet->decrement('balance', $amount);

            // 3. FIFO Consumption from Deposits
            $amountToConsume = $amount;
            $deposits = WalletDeposit::where('wallet_id', $wallet->id)
                ->where('status', 'active')
                ->where('current_value', '>', 0)
                ->orderBy('created_at', 'asc')
                ->lockForUpdate()
                ->get();

            foreach ($deposits as $deposit) {
                if ($amountToConsume <= 0) break;

                if ($deposit->current_value >= $amountToConsume) {
                    $deposit->decrement('current_value', $amountToConsume);
                    if ($deposit->fresh()->current_value <= 0) {
                        $deposit->update(['status' => 'fully_consumed']);
                    }
                    $amountToConsume = 0;
                } else {
                    $consumed = $deposit->current_value;
                    $deposit->update(['current_value' => 0, 'status' => 'fully_consumed']);
                    $amountToConsume -= $consumed;
                }
            }

            return $transaction;
        });
    }




//For FullCharge The Wallet With Money
    public function deposit(User $user, float $amount, string $type = 'deposit', $referenceModel = null, string $description = '', $stripeFee = 0, $stripePaymentId = null)
    {
        return DB::transaction(function () use ($user, $amount, $type, $referenceModel, $description, $stripeFee, $stripePaymentId) {
            $wallet = $user->wallet()->lockForUpdate()->first();

            $wallet->increment('balance', $amount);

            $transaction = $wallet->transactions()->create([
                'type' => $type,
                'amount' => $amount,
                'balance_after' => $wallet->balance,
                'reference_id' => $referenceModel ? $referenceModel->id : null,
                'reference_type' => $referenceModel ? get_class($referenceModel) : null,
                'description' => $description,
                'stripe_fee' => $stripeFee,
                'stripe_payment_id' => $stripePaymentId,
            ]);

            // Create Inventory Record
            WalletDeposit::create([
                'wallet_id' => $wallet->id,
                'stripe_payment_id' => $stripePaymentId,
                'amount' => $amount,
                'current_value' => $amount,
                'status' => 'active',
                'expires_at' => $stripePaymentId ? now()->addDays(180) : null,
            ]);

            return $transaction;
        });
    }







//For FullWithdraw The Money From The Wallet
    public function withdraw(User $user, float $amount)
    {
        // 1. Convert to Cents immediately to avoid float issues
        $amountInCents = (int) round($amount * 100);

        return DB::transaction(function () use ($user, $amountInCents) {
            // Fix: Check if wallet exists
            $wallet = $user->wallet()->lockForUpdate()->first();
            if (!$wallet) {
                return $this->setError('Wallet not found.');
            }

            // Compare using integers
            $currentBalanceCents = (int) round($wallet->balance * 100);
            if ($currentBalanceCents < $amountInCents) {
                return $this->setError('Insufficient wallet balance.');
            }

            $cutoffDate = now()->subDays(179);

            $deposits = WalletDeposit::where('wallet_id', $wallet->id)
                ->where('current_value', '>', 0)
                ->where('created_at', '>', $cutoffDate)
                ->whereNotNull('stripe_payment_id')
                ->orderBy('created_at', 'asc') // FIFO is good for expiration
                ->lockForUpdate()
                ->get();

            $remainingCentsToWithdraw = $amountInCents;
            $refundsProcessed = [];

            Stripe::setApiKey(env('STRIPE_SECRET'));

            foreach ($deposits as $deposit) {
                if ($remainingCentsToWithdraw <= 0) break;

                // Calculate amount for this deposit (in cents)
                $depositValueCents = (int) round($deposit->current_value * 100);
                $refundAmountCents = min($depositValueCents, $remainingCentsToWithdraw);

                try {
                    // OPTIMIZATION: Try refund directly. Don't check PaymentIntent unless it fails.
                    $refund = \Stripe\Refund::create([
                        'payment_intent' => $deposit->stripe_payment_id,
                        'amount' => $refundAmountCents,
                    ]);

                    // If success, update local state
                    $refundsProcessed[] = $refund->id;
                } catch (\Stripe\Exception\InvalidRequestException $e) {
                    // Catch specific Stripe errors
                    $body = $e->getJsonBody();
                    $err  = $body['error'];

                    // Handle "Charge already refunded" or similar logic mismatch
                    if (str_contains($err['message'] ?? '', 'refunded')) {
                        // Mark as consumed locally so we don't try again
                        $deposit->update(['current_value' => 0, 'status' => 'fully_consumed']);
                        continue; // Skip to next deposit
                    }

                    // Other errors (e.g. expired, disputed) -> Log and skip
                    Log::error("Stripe Refund Failed: " . $e->getMessage());
                    continue;
                } catch (\Exception $e) {
                    continue;
                }

                // Update Ledger (Database)
                // We convert back to float for DB storage only
                $refundedAmountFloat = $refundAmountCents / 100.0;

                $deposit->decrement('current_value', $refundedAmountFloat);

                // Check precision-safe zero
                if ($deposit->fresh()->current_value < 0.01) {
                    $deposit->update(['status' => 'refunded', 'current_value' => 0]);
                }

                // Update Wallet Balance
                $wallet->decrement('balance', $refundedAmountFloat);

                $wallet->transactions()->create([
                    'type' => 'refund', // Changed from 'refund' to be clearer for user
                    'amount' => -$refundedAmountFloat,
                    'balance_after' => $wallet->fresh()->balance,
                    'description' => "Withdrawal ref: " . substr($deposit->stripe_payment_id, -8),
                    'stripe_payment_id' => $refund->id, // Store Refund ID, not Payment ID
                ]);

                $remainingCentsToWithdraw -= $refundAmountCents;
            }

            // Final Check
            if (empty($refundsProcessed) && $remainingCentsToWithdraw == $amountInCents) {
                return $this->setError('Could not process withdrawal. Please contact support.');
            }

            // return partial success or full success info
            return [
                'status' => 'success',
                'refunded_ids' => $refundsProcessed,
                'amount_refunded' => ($amountInCents - $remainingCentsToWithdraw) / 100
            ];
        });
    }


}
