<?php

namespace Modules\Payment\Services;

use Stripe\StripeClient;
use Modules\User\Models\User;
use Exception;

class StripeConnectService
{
    protected $stripe;

    public function __construct()
    {
        $this->stripe = new StripeClient(env('STRIPE_SECRET'));
    }

    /**
     * Create a Stripe Express Account with Automatic  Payouts.
     *
     * @param User $user
     * @return \Stripe\Account
     */
    public function createExpressAccount(User $user)
    {
        if ($user->stripe_account_id) {
            return $this->stripe->accounts->retrieve($user->stripe_account_id);
        }

        $account = $this->stripe->accounts->create([
            'type' => 'express',
            'country' => 'GB', // Defaulting to GB as per request context
            'email' => $user->email,
            'capabilities' => [
                'transfers' => ['requested' => true],
            ],
            'settings' => [
                'payouts' => [
                    'schedule' => [
                        'interval' => 'weekly',
                        'weekly_anchor' => 'monday',
                    ],
                ],
            ],
        ]);

        // Save the account ID to the user
        $user->forceFill(['stripe_account_id' => $account->id])->save();

        return $account;
    }

    /**
     * Generate an Account Link for Onboarding.
     *
     * @param User $user
     * @return string
     */
    public function getAccountLink(User $user)
    {
        if (!$user->stripe_account_id) {
            $this->createExpressAccount($user);
        }

        $accountLink = $this->stripe->accountLinks->create([
            'account' => $user->stripe_account_id,
            'refresh_url' => route('api.payment.connect.refresh'), // Route needs to be defined
            'return_url' => route('api.payment.connect.return'),   // Route needs to be defined
            'type' => 'account_onboarding',
        ]);

        return $accountLink->url;
    }




    /**
     * Transfer funds from Platform to Connected Account.
     *
     * @param User $user
     * @param float $amount Amount in major currency unit (e.g. 50.00)
     * @param string $currency
     * @return \Stripe\Transfer
     * @throws Exception
     */
    public function transferFunds(User $user, float $amount, string $currency = 'gbp')
    {
        if (!$user->stripe_account_id) {
            throw new Exception("User does not have a Stripe Connect account.");
        }

        return $this->stripe->transfers->create([
            'amount' => (int)($amount * 100), // Convert to cents/pence
            'currency' => $currency,
            'destination' => $user->stripe_account_id,
        ]);
    }
}
