<?php

namespace Modules\Payment\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Modules\Payment\Services\StripeConnectService;

class ConnectController extends Controller
{
    protected $connectService;

    public function __construct(StripeConnectService $connectService)
    {
        $this->connectService = $connectService;
    }

    /**
     * Get Onboarding Link for the authenticated user.
     * POST /api/payment/connect/onboard
     */
    public function onboard(Request $request)
    {
        try {

            $user = auth('api')->user(); 
            
            $url = $this->connectService->getAccountLink($user);

            return response()->json([
                'status' => 'success',
                'url' => $url,
                'message' => 'Onboarding link generated successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Simulate a transfer to the connected account (Test only).
     * POST /api/payment/connect/test-transfer
     */
    public function testTransfer(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1',
        ]);

        try {
            $user = auth('api')->user();
            $amount = $request->amount;

            $transfer = $this->connectService->transferFunds($user, $amount);

            return response()->json([
                'status' => 'success',
                'data' => $transfer,
                'message' => "Successfully transferred {$amount} GBP to connected account."
            ]);
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Add funds to Platform Account (Testing Only)
     * POST /api/payment/connect/add-platform-funds
     */
    public function addPlatformFunds(Request $request)
    {
        try {
            $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET'));

            $charge = $stripe->charges->create([
                'amount' => 100000, // £1000.00
                'currency' => 'gbp',
                'source' => 'tok_bypassPending', // Magic token for instant available balance
                'description' => 'Adding funds for testing transfers',
            ]);

            return response()->json([
                'status' => 'success',
                'data' => $charge,
                'message' => 'Successfully added £1000 to Platform Balance.'
            ]);
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => $e->getMessage()], 500);
        }
    }

}
