<?php

namespace Modules\Common\Controllers;

use App\Enum\MatchType;
use App\Enum\NotifyType;
use App\Helpers\ApiResponse;
use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Modules\Common\Models\NotificationAction;
use Modules\Common\Models\Setting;
use Modules\Common\Resources\Notification\NotificationCollection;
use Modules\Common\Resources\Notification\NotificationResource;
use Modules\Common\Resources\ScheduleDayResource;
use Modules\Match\Models\Matche;
use Modules\Match\Resources\MatcheResource;
use Modules\Sliders\Models\Slider;
use Modules\Sliders\Resources\SliderResource;

class ApiController extends Controller
{
    public function home()
    {
        $sliders = Slider::active()->orderBy('sort')->get();
        $matches = Matche::where('type', MatchType::SINGLE)->filter()->when(request()->header('lat') && request()->header('lng'), fn($q) => $q->nearest(request()->header('lat'), request()->header('lng')))->active()->inTime()->with('playground')->withCount('subscribers')->get();
        return ApiResponse::loaded([
            'sliders' => SliderResource::collection($sliders),
            'matches' => MatcheResource::collection($matches),
        ]);
    }
    public function groupMatches()
    {
        $matches = Matche::where('type', MatchType::GROUP)->filter()->when(request()->header('lat') && request()->header('lng'), fn($q) => $q->nearest(request()->header('lat'), request()->header('lng')))->active()->inTime()->with('playground')->get();
        return ApiResponse::loaded([
            'matches' => MatcheResource::collection($matches),
        ]);
    }
    public function schedules()
    {
        $days_count = 14;
        $startDate = Carbon::now();
        $endDate = Carbon::now()->addDays((int) $days_count);
        $dateRange = CarbonPeriod::create($startDate, $endDate);
        return ApiResponse::loaded([
            'days' => ScheduleDayResource::collection($dateRange->toArray()),
        ]);
    }
    public function notifications()
    {
        $user = auth('api')->user();
        throw_if(!$user, ValidationException::withMessages(['msg' => __('auth failed')]));
        $notifications = $user->allNotifications()->latest()->paginate(20);
        return ApiResponse::loaded([
            'notifications' => NotificationResource::collection($notifications)
        ]);
    }
    public function deleteNotification($uuid)
    {
        $user = auth('api')->user();
        throw_if(!$user, ValidationException::withMessages(['msg' => __('auth failed')]));
        $notification = $user->allNotifications()->where('id', $uuid)->firstOrFail();
        if ($notification->notifiable_id > 0) {
            $notification->delete();
        } else {
            NotificationAction::create([
                'user_id' => auth('api')->id(),
                'notification_id' => $uuid,
                'type' => NotifyType::DELETE,
            ]);
        }
        return ApiResponse::deleted();
    }

    public function notifyStatus(Request $request)
    {
        $user = auth('api')->user();
        $user->update([
            'notify' => !$user->notify
        ]);
        return ApiResponse::updated();
    }

    public function settings()
    {
        $settings = Setting::all()->map(function ($setting) {
            $val = $setting->value;
            if (is_array($val) || is_object($val)) {
                $val = (array) $val;
                $val = $val['all'] ?? ($val[app()->getLocale()] ?? array_values($val)[0] ?? '');
            }

            $data = [
                'id' => $setting->id,
                'key' => $setting->key,
                'type' => $setting->type,
                'value' => $val,
            ];

            if ($setting->type == 'contacts') {
                $data['image'] = $setting->getFirstMediaUrl('icon');
            } elseif (in_array($setting->key, ['logo', 'favicon'])) {
                $data['image'] = $setting->value;
            }

            return $data;
        });

        return ApiResponse::loaded([
            'data' => $settings
        ]);
    }
}
