<?php

namespace App\Http\Requests\Admin\Matches;

use Carbon\Carbon;
use Illuminate\Foundation\Http\FormRequest;

class Store extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    protected function passedValidation()
    {
        $this->merge([
            'end_time' => Carbon::parse($this->end_time)->format('H:i:s'),
            'start_time' => Carbon::parse($this->start_time)->format('H:i:s'),
        ]);
    }

    public function rules(): array
    {
        return [
            'playground_id' => 'required|exists:playgrounds,id,deleted_at,NULL',
            'organizer_user_id' => 'required|exists:users,id,deleted_at,NULL',
            'players_per_side' => 'required|integer|numeric|min:1|max:30',
            'name' => 'required|string',
            'About' => 'nullable|string',
            'date' => 'required|date|after:yesterday',
            'start_time' => 'required',
            'end_time' => 'required|after:start_time',
            'price' => 'required|numeric|min:0',
            'gender' => 'required|in:male,female,co_ed',
            'status' => 'nullable|boolean',
        ];
    }

    protected function prepareForValidation()
    {
        $this->merge([
            'revision' => false
        ]);
    }

    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            $startTime = Carbon::parse($this->start_time)->format('H:i:s');
            $endTime = Carbon::parse($this->end_time)->format('H:i:s');
            $date = Carbon::parse($this->date)->format('Y-m-d');

            $exists = \Modules\Match\Models\Matche::where('playground_id', $this->playground_id)
                ->where('date', $date)
                ->where(function ($query) use ($startTime, $endTime) {
                    $query->where(function ($q) use ($startTime, $endTime) {
                        $q->where('start_time', '>=', $startTime)
                            ->where('start_time', '<', $endTime);
                    })->orWhere(function ($q) use ($startTime, $endTime) {
                        $q->where('end_time', '>', $startTime)
                            ->where('end_time', '<=', $endTime);
                    })->orWhere(function ($q) use ($startTime, $endTime) {
                        $q->where('start_time', '<=', $startTime)
                            ->where('end_time', '>=', $endTime);
                    });
                })
                ->when($this->route('match') ?? $this->input('id'), function ($q, $id) {
                    $q->where('id', '!=', $id);
                })
                ->exists();

            if ($exists) {
                $validator->errors()->add('start_time', __('There is already a match scheduled in this playground at this time.'));
            }
        });
    }
}
