<?php

namespace App\Console\Commands;

use App\Notifications\PostMatchReviewNotification;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Notification;
use Modules\Match\Models\Matche;

class NotifyPostMatchReview extends Command
{
    protected $signature = 'notify:post-match-review';
    protected $description = 'Notify users to review their match after it ends';

    public function handle()
    {
        // Check matches ended between 1 and 2 hours ago
        $startTime = Carbon::now()->subHours(2);
        $endTime = Carbon::now()->subHour();

        $matches = Matche::whereDate('date', Carbon::now()->format('Y-m-d')) // Optimization: only check today's matches (or yesterday's if around midnight)
            ->where(function ($query) use ($startTime, $endTime) {
                // Combine date and end_time to compare
                // Since end_time is just time, we need to be careful.
                // Assuming matches don't span days for now or date is the start date.
                // Actually, Matche has 'date' and 'end_time'.
                // We need to construct the full datetime.
            })
            ->active()
            ->with(['teamPlayers', 'playground'])
            ->get();

        // Better approach for SQL:
        // We can iterate all matches of today and yesterday and check in PHP to be safe with date/time combination
        $matches = Matche::whereIn('date', [Carbon::now()->format('Y-m-d'), Carbon::now()->subDay()->format('Y-m-d')])
            ->active()
            ->with(['teamPlayers', 'playground'])
            ->get();

        foreach ($matches as $match) {
            $matchEndTime = Carbon::parse($match->date . ' ' . $match->end_time);

            if ($matchEndTime->between($startTime, $endTime)) {
                $players = $match->teamPlayers->filter(function ($user) {
                    return $user->wantsNotification('Post-Match Review');
                });

                if ($players->count() > 0) {
                    Notification::send($players, new PostMatchReviewNotification($match));
                    $this->info("Sent post-match review alert for match ID: {$match->id}");
                }
            }
        }
    }
}
