<?php

namespace Modules\Playground\Services;

use App\Services\Services;
use Modules\Playground\Models\Playground;
use Modules\Playground\Requests\UpdatePlaygroundRequest;

class PlaygroundService extends Services
{
    private $columns = [
        'id' => 'id',
        'name' => 'Name',
        'address' => 'Address',
        'city' => 'City',
        'latitude' => 'Latitude',
        'longitude' => 'Longitude',
        'category_id' => 'Category_ID',
        'owner_id' => 'Owner_ID',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
        'accessibility' => 'Accessibility',
    ];

    public function paginate()
    {
        $column = request('sort_by', 'id');
        $order = request('sort_order') ?: 'desc';
        $name = request('name', null);
        $address = request('address', null);
        $city = request('city', null);
        $latitude = request('latitude', null);
        $longitude = request('longitude', null);
        $category_id = config('context.category_id') ?? request('category_id', null);
        $user_id = request('user_id', null);
        $created_at = request('created_at', null);
        $updated_at = request('updated_at', null);
        $accessibility = request('accessibility', null);
        $disability_parking = request('disability_parking', null);
        $disability_activity_areas = request('disability_activity_areas', null);
        $disability_social_areas = request('disability_social_areas', null);
        $disability_spectator_areas = request('disability_spectator_areas', null);
        $disability_changing_facilities = request('disability_changing_facilities', null);
        $disability_toilets = request('disability_toilets', null);
        $disability_finding_and_reaching_the_entrance = request('disability_finding_and_reaching_the_entrance', null);
        $disability_reception_area = request('disability_reception_area', null);
        $disability_doorways = request('disability_doorways', null);
        $disability_emergency_exits = request('disability_emergency_exits', null);
        $status = request('status', null);

        $query = Playground::query()->with(['category', 'user'])
            ->when($status, function ($q, $status) {
                $q->where('status', $status);
            }, function ($q) {
                $q->where('status', 'active');
            })
            ->when($name, function ($q, $name) {
                $q->where('name', 'LIKE', "%$name%");
            })->when($address, function ($q, $address) {
                $q->where('address', 'LIKE', "%$address%");
            })
            ->when($city, function ($q, $city) {
                $q->where('city', 'LIKE', "%$city%");
            })
            ->when($latitude, function ($q, $latitude) {
                $q->where('latitude', 'LIKE', "%$latitude%");
            })
            ->when($longitude, function ($q, $longitude) {
                $q->where('longitude', 'LIKE', "%$longitude%");
            })
            ->when($category_id, function ($q, $category_id) {
                $q->where('Category_ID', $category_id);
            })
            ->when($user_id, function ($q, $user_id) {
                $q->where('owner_id', 'LIKE', "%$user_id%");
            })
            ->when($created_at, function ($q, $created_at) {
                $q->where('created_at', 'LIKE', "%$created_at%");
            })
            ->when($updated_at, function ($q, $updated_at) {
                $q->where('updated_at', 'LIKE', "%$updated_at%");
            })
            ->when($accessibility, function ($q, $accessibility) {
                $q->where('accessibility', 'LIKE', "%$accessibility%");
            })
            ->when(request()->has('disability_parking'), function ($q) {
                $q->where('Disability_Parking', request()->boolean('disability_parking'));
            })
            ->when(request()->has('disability_activity_areas'), function ($q) {
                $q->where('Disability_Activity_Areas', request()->boolean('disability_activity_areas'));
            })
            ->when(request()->has('disability_social_areas'), function ($q) {
                $q->where('Disability_Social_Areas', request()->boolean('disability_social_areas'));
            })
            ->when(request()->has('disability_spectator_areas'), function ($q) {
                $q->where('Disability_Spectator_Areas', request()->boolean('disability_spectator_areas'));
            })
            ->when(request()->has('disability_changing_facilities'), function ($q) {
                $q->where('Disability_Changing_Facilities', request()->boolean('disability_changing_facilities'));
            })
            ->when(request()->has('disability_toilets'), function ($q) {
                $q->where('Disability_Toilets', request()->boolean('disability_toilets'));
            })
            ->when(request()->has('disability_finding_and_reaching_the_entrance'), function ($q) {
                $q->where('Disability_Finding_and_Reaching_the_Entrance', request()->boolean('disability_finding_and_reaching_the_entrance'));
            })
            ->when(request()->has('disability_reception_area'), function ($q) {
                $q->where('Disability_Reception_Area', request()->boolean('disability_reception_area'));
            })
            ->when(request()->has('disability_doorways'), function ($q) {
                $q->where('Disability_Doorways', request()->boolean('disability_doorways'));
            })
            ->when(request()->has('disability_emergency_exits'), function ($q) {
                $q->where('Disability_Emergency_Exits', request()->boolean('disability_emergency_exits'));
            });

        if ($column && array_key_exists($column, $this->columns)) {
            $query->orderBy($this->columns[$column], $order);
        }

        return $query->paginate(request('limit', 15));
    }

    public function getAllPlaygrounds()
    {
        return Playground::active()->get();
    }







    public function store(array $data)
    {
        $mappedData = [
            'Name' => $data['name'],
            'Owner' => auth("api")->user()->FirstName . ' ' . auth("api")->user()->LastName,
            'Owner_ID' => auth("api")->id(),
            'Category_ID' => $data['Category_ID'] ?? config('context.category_id'),
            'Address' => $data['address'],
            'City' => $data['city'],
            'Telephone' => $data['telephone'],
            'Email' => $data['email'],
            'Grass_Type' => $data['grass_type'],
            'Accessibility' => $data['accessibility'],
            'Latitude' => $data['latitude'],
            'Longitude' => $data['longitude'],
            'status' => $this->determineStatus(),
            'Disability_Parking' => $data['disability_parking'] ?? false,
            'Disability_Activity_Areas' => $data['disability_activity_areas'] ?? false,
            'Disability_Social_Areas' => $data['disability_social_areas'] ?? false,
            'Disability_Spectator_Areas' => $data['disability_spectator_areas'] ?? false,
            'Disability_Changing_Facilities' => $data['disability_changing_facilities'] ?? false,
            'Disability_Toilets' => $data['disability_toilets'] ?? false,
            'Disability_Finding_and_Reaching_the_Entrance' => $data['disability_finding_and_reaching_the_entrance'] ?? false,
            'Disability_Reception_Area' => $data['disability_reception_area'] ?? false,
            'Disability_Doorways' => $data['disability_doorways'] ?? false,
            'Disability_Emergency_Exits' => $data['disability_emergency_exits'] ?? false,
            'slots' => $data['slots'] ?? [],
            'media' => $data['media'] ?? null,
        ];




        return Playground::create($mappedData);
    }



    public function update(array $data, $id)
    {
        $playground = Playground::find($id);
        if (!$playground) {

            $this->setError(__('messages.Playground not found'));

            return false;
        }
        $playground->update($data);

        return $playground;
    }

    public function destroy($id)
    {
        $playground = Playground::find($id);
        if (!$playground) {
            $this->setError(__('messages.Playground not found'));
            return false;
        }
        $playground->delete();

        return $playground;
    }

    public function show($id)
    {
        $query = Playground::with(['media', 'category', 'user', 'reviews']);

        if (request()->has(['lat', 'lng'])) {
            $query->withDistance(request('lat'), request('lng'));
        }

        $playground = $query->find($id);

        if (!$playground) {
            $this->setError(__('messages.Playground not found'));
            return false;
        }
        return $playground;
    }


    public function Deactivite($id)
    {
        $playground = Playground::find($id);
        if (!$playground) {
            $this->setError(__('messages.Playground not found'));
            return false;
        }
        $playground->update(['status' => false]);

        return $playground;
    }






    public function getNearest($lat, $lng)
    {
        $query = Playground::active()->with(['user']);

        if ($catId = config('context.category_id')) {
            $query->where('Category_ID', $catId);
        }

        $playgrounds = $query->nearest($lat, $lng)->take(5)->get();

        if ($playgrounds->isEmpty()) {
            $fallbackQuery = Playground::active()->with(['user']);
            if ($catId) {
                $fallbackQuery->where('Category_ID', $catId);
            }
            $playgrounds = $fallbackQuery->withDistance($lat, $lng)->latest()->take(5)->get();
        }

        return $playgrounds;
    }

    public function myplaygrounds()
    {
        return Playground::where('Owner_ID', auth("api")->id())
            ->where('status', 'pending')
            ->with(['media', 'category', 'user', 'reviews'])
            ->latest()
            ->get();
    }

    private function determineStatus()
    {
        $user = auth("api")->user();

        // Player (Role ID 2) always creates active playgrounds
        if ($user->role_id == 2) {
            return 'active';
        }

        // Playground Master (Role ID 1)
        if ($user->role_id == 1) {
            $hasActivePlayground = Playground::where('Owner_ID', $user->id)
                ->where('status', 'active')
                ->exists();

            if ($hasActivePlayground) {
                return 'active';
            }
        }

        return 'pending';
    }
}
