<?php

namespace App\Traits;

use Illuminate\Http\UploadedFile;
use Spatie\MediaLibrary\InteractsWithMedia;

trait HasUnifiedMedia
{
    use InteractsWithMedia;

    public function setMediaAttribute($files)
    {
        // 1. Handle Single File
        if ($files instanceof UploadedFile) {
            $this->processFile($files);
            return;
        }

        // 2. Handle Array
        if (is_array($files)) {
            foreach ($files as $key => $file) {
                // If key is a string (e.g., 'cover'), use it as collection
                // Otherwise (numeric index), it's auto-detected
                $collection = is_string($key) ? $key : null;

                // Recursively handle nested arrays (e.g. media[gallery][])
                if (is_array($file)) {
                    foreach ($file as $subFile) {
                        $this->processFile($subFile, $collection);
                    }
                } else {
                    $this->processFile($file, $collection);
                }
            }
        }
    }

    protected function processFile($file, $collection = null)
    {
        if (!$file instanceof UploadedFile) return;

        // Auto-detect collection if not provided
        if (!$collection) {
            $mime = $file->getMimeType();
            if (str_contains($mime, 'image')) {
                $collection = 'images';
            } elseif (str_contains($mime, 'video')) {
                $collection = 'video';
            } else {
                $collection = 'default';
            }
        }

        // Add to collection
        $this->addMedia($file)->toMediaCollection($collection);
    }

    public function getMediaListAttribute()
    {
        return $this->getMedia('*')->map(function ($item) {
            return [
                'id' => $item->id,
                'url' => url($item->getUrl()),
                'type' => str_contains($item->mime_type, 'video') ? 'video' : 'image',
                'collection' => $item->collection_name,
            ];
        });
    }

    // Backward compatibility accessors (optional but good to have)
    public function getImageAttribute()
    {
        $url = $this->getFirstOrDefaultMediaUrl('images');
        return $url ? url($url) : null;
    }

    public function getVideoAttribute()
    {
        $url = $this->getFirstOrDefaultMediaUrl('video');
        return $url ? url($url) : null;
    }

    public function getVisualAttribute()
    {
        $item = $this->getMedia('*')->first();
        if (!$item) return '';

        $url = url($item->getUrl());
        if (str_contains($item->mime_type, 'image')) {
            return '<a href="' . $url . '" target="_blank"><img src="' . $url . '" style="width: 50px; height: 50px; object-fit: cover; border-radius: 4px;"></a>';
        } elseif (str_contains($item->mime_type, 'video')) {
            return '<a href="' . $url . '" target="_blank"><video src="' . $url . '" style="width: 50px; height: 50px; object-fit: cover; border-radius: 4px;"></video></a>';
        }
        return '';
    }
}
